import React from 'react';

import {
    TBookLegalInfo,
    IBookOfferNightlyCharges,
    IBookOfferPrice,
} from 'server/api/HotelsBookAPI/types/IBookOffer';
import {IWithClassName} from 'types/withClassName';
import {IPromoCodeApplicationResult} from 'types/hotels/book/IApplyPromoCodes';
import {
    EYandexPlusApplicationMode,
    IAppliedPromoCampaigns,
} from 'types/hotels/offer/IHotelOffer';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {CHAR_MINUS, CHAR_THIN_SPACE} from 'utilities/strings/charCodes';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import IPrice from 'utilities/currency/PriceInterface';

import * as i18nBlock from 'i18n/account-OrderHotels-Price';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Price from 'components/Price/Price';
import Separator from 'components/Separator/Separator';
import BookLicenseAgreement from 'projects/hotels/components/BookLicenseAgreement/BookLicenseAgreement';
import Box from 'components/Box/Box';
import PartialPrice from 'components/PartialPrice/PartialPrice';
import PlusInfo from 'projects/account/pages/Order/components/OrderHotels/components/OrderHotelMainInfo/components/OrderHotelsPrice/components/PlusInfo/PlusInfo';
import DetailItem from 'projects/account/pages/Order/components/OrderHotels/components/DetailItem/DetailItem';

import cx from './Details.scss';

interface IDetailsProps extends IWithClassName, IWithQaAttributes {
    amountPaid: IPrice | undefined;
    totalPrice: IPrice;
    nights: IBookOfferNightlyCharges[];
    hotelPrice: IBookOfferPrice;
    discount: IBookOfferPrice | undefined;
    taxesAndFeesSum: IBookOfferPrice | undefined;
    codeApplicationResults: IPromoCodeApplicationResult[];
    legalInfo: TBookLegalInfo;
    appliedPromoCampaigns: IAppliedPromoCampaigns | undefined;
}

const Details: React.FC<IDetailsProps> = props => {
    const {
        className,
        nights,
        amountPaid,
        totalPrice,
        hotelPrice,
        discount,
        taxesAndFeesSum,
        codeApplicationResults,
        legalInfo,
        appliedPromoCampaigns,
    } = props;

    const deviceType = useDeviceType();

    return (
        <div className={cx(className, 'root', deviceMods('root', deviceType))}>
            <Flex flexDirection="column" between={3}>
                <DetailItem
                    title={i18nBlock.nights({
                        nights: nights.length,
                    })}
                    price={
                        <Price
                            currency={hotelPrice.currency}
                            value={hotelPrice.amount}
                        />
                    }
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'totalNights',
                    })}
                />
                {taxesAndFeesSum && (
                    <DetailItem
                        title={i18nBlock.taxes()}
                        price={
                            <Price
                                currency={taxesAndFeesSum.currency}
                                value={taxesAndFeesSum.amount}
                            />
                        }
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'taxesAndFeesSum',
                        })}
                    />
                )}
                <PlusInfo
                    direction="row"
                    appliedPromoCampaigns={appliedPromoCampaigns}
                    yandexPlusMode={EYandexPlusApplicationMode.WITHDRAW}
                />
                {discount && (
                    <DetailItem
                        title={i18nBlock.discount()}
                        price={
                            <Price
                                currency={discount.currency}
                                value={discount.amount}
                                color="alert"
                                prefix={CHAR_MINUS}
                                prefixDelimiter={CHAR_THIN_SPACE}
                            />
                        }
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'discount',
                        })}
                    />
                )}
                {codeApplicationResults.map(
                    ({discountAmount, code}, index) =>
                        discountAmount && (
                            <DetailItem
                                key={code}
                                title={i18nBlock.promoCodeDiscount()}
                                price={
                                    <Price
                                        value={discountAmount.value}
                                        currency={discountAmount.currency}
                                        color="alert"
                                        prefix={CHAR_MINUS}
                                        prefixDelimiter={CHAR_THIN_SPACE}
                                        {...prepareQaAttributes({
                                            key: index,
                                            current: 'promoCodeDiscount',
                                        })}
                                    />
                                }
                                additional={
                                    <Text
                                        size="xs"
                                        color="secondary"
                                        {...prepareQaAttributes({
                                            key: index,
                                            current: 'promoCode',
                                        })}
                                    >
                                        {code}
                                    </Text>
                                }
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'promoCode',
                                    key: code,
                                })}
                            />
                        ),
                )}
            </Flex>

            <Separator margin={4} />

            <Flex flexDirection="column" between={3}>
                <DetailItem
                    title={
                        <Text size="inherit" weight="bold">
                            {i18nBlock.paidTitle()}
                        </Text>
                    }
                    price={
                        <Text size="inherit" weight="bold">
                            <PartialPrice
                                price={amountPaid}
                                totalPrice={totalPrice}
                            />
                        </Text>
                    }
                />
                <PlusInfo
                    direction="row"
                    appliedPromoCampaigns={appliedPromoCampaigns}
                    yandexPlusMode={EYandexPlusApplicationMode.TOPUP}
                />
            </Flex>

            <Box above={8}>
                <Text size="xs" color="secondary">
                    <BookLicenseAgreement
                        isBookStep={false}
                        legalInfo={legalInfo}
                    />
                </Text>
            </Box>
        </div>
    );
};

export default Details;
