import React, {useCallback, useEffect, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {
    CONFIRMED,
    PAYMENT_FAILED,
    AWAITS_PAYMENT,
} from 'projects/hotels/constants/hotelsBookingStatuses';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {IOrderInfoRequestPayload} from 'reducers/hotels/bookAndPayPage/orderInfo/types';
import {IWithClassName} from 'types/withClassName';
import {
    EDownloadActionTypes,
    ESearchByOrderId,
} from 'server/api/HotelsBookAPI/types/IDownloadVoucher';

import {
    fetchRefundableInfoActions,
    setRefundModalVisible,
    startCancelOrderActions,
} from 'reducers/hotels/bookAndPayPage/orderCancelAndRefundableInfo/actions';

import orderCancelAndRefundableInfoSelector from 'selectors/hotels/book/orderCancelAndRefundableInfoSelector';

import {deviceModMobile, deviceMods} from 'utilities/stylesUtils';
import {getHotelsVoucherUrl} from 'projects/hotels/utilities/getHotelsVoucherUrl/getHotelsVoucherUrl';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {reachGoal} from 'utilities/metrika';
import {useHotelVoucherStatus} from 'utilities/hooks/useHotelVoucherStatus';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nCancelOrderBlock from 'i18n/hotels-CancelOrder';
import * as i18nBlock from 'i18n/hotels-BookSuccessPage';
import * as i18nHappyPageBlock from 'i18n/happyPage';

import OrderActions, {
    EOrderActionsSource,
    IOrderActionsCancel,
    IOrderActionsDownload,
    IOrderActionsPrint,
} from 'components/OrderActions/OrderActions';
import Modal from 'components/Modal/Modal';
import OrderHotelsCancelModal from 'projects/account/pages/Order/components/OrderHotels/components/OrderHotelsActions/components/OrderHotelsCancelModal';
import Flex from 'components/Flex/Flex';
import Spinner from 'components/Spinner/Spinner';
import YandexMessengerButton from 'components/YandexMessenger/components/YandexMessengerButton/YandexMessengerButton';
import MessageOutlineIcon from 'icons/16/MessageOutline';

import cx from './OrderHotelsActions.scss';

interface IOrderHotelsActionsProps extends IWithClassName, IWithQaAttributes {
    orderId: string;
    status: string;
    yandexOrderId: string;
    isCancelEnabled: boolean;
    inactive: boolean;
    startOrderPolling(payload: IOrderInfoRequestPayload): void;
}

const STATUS_LIST_WITH_ACTION_BUTTONS = [
    CONFIRMED,
    PAYMENT_FAILED,
    AWAITS_PAYMENT,
];

const OrderHotelsActions: React.FC<IOrderHotelsActionsProps> = props => {
    const {
        className,
        orderId,
        yandexOrderId,
        isCancelEnabled,
        startOrderPolling,
        status,
        inactive,
    } = props;

    const deviceType = useDeviceType();
    const dispatch = useDispatch();

    const {orderCancelAndRefundableInfo} = useSelector(
        orderCancelAndRefundableInfoSelector,
    );

    const {refundableInfo, cancelOrderInfo} = orderCancelAndRefundableInfo;

    const setRefundModalVisibility = useCallback(
        (visible: boolean) => dispatch(setRefundModalVisible(visible)),
        [dispatch],
    );
    const closeRefundModal = useCallback(
        () => setRefundModalVisibility(false),
        [setRefundModalVisibility],
    );

    const {isReady: isVoucherReady, isLoading: isVoucherStatusLoading} =
        useHotelVoucherStatus(orderId);

    const fetchRefundableInfo = useCallback(
        (id: string) => {
            return dispatch(fetchRefundableInfoActions.request(id));
        },
        [dispatch],
    );

    const handleFetchRefundableInfoClick = useCallback(() => {
        setRefundModalVisibility(true);
        fetchRefundableInfo(orderId);

        reachGoal(EHotelsGoal.ORDER_CANCEL_BUTTON);
    }, [orderId, setRefundModalVisibility, fetchRefundableInfo]);

    const handleCancelOrder = useCallback(
        payload => {
            return dispatch(startCancelOrderActions.request(payload));
        },
        [dispatch],
    );

    const renderIcon = useCallback(
        (cn?: string, defaultIcon?: JSX.Element): JSX.Element | undefined => {
            if (deviceType.isMobile) {
                return undefined;
            }

            if (isVoucherStatusLoading) {
                return <Spinner className={cn} size="xxs" />;
            }

            return defaultIcon;
        },
        [deviceType.isMobile, isVoucherStatusLoading],
    );

    const printInfo = useMemo((): IOrderActionsPrint => {
        return {
            title: i18nBlock.printOrderInfoShort(),
            isDisabled: !isVoucherReady,
            renderIcon,
            url: getHotelsVoucherUrl(yandexOrderId, {
                actionType: EDownloadActionTypes.PRINT,
                searchBy: ESearchByOrderId.PRETTY_ORDER_ID,
            }),
        };
    }, [isVoucherReady, renderIcon, yandexOrderId]);

    const downloadInfo = useMemo((): IOrderActionsDownload => {
        return {
            title: i18nBlock.downloadVoucherShort(),
            url: getHotelsVoucherUrl(yandexOrderId, {
                actionType: EDownloadActionTypes.DOWNLOAD,
                searchBy: ESearchByOrderId.PRETTY_ORDER_ID,
            }),
            isDisabled: !isVoucherReady,
            renderIcon,
        };
    }, [isVoucherReady, renderIcon, yandexOrderId]);

    const cancelInfo = useMemo((): IOrderActionsCancel | undefined => {
        if (!isCancelEnabled) {
            return undefined;
        }

        return {
            title: i18nCancelOrderBlock.cancelButton2(),
            isDisabled: refundableInfo.isError,
            renderIcon,
            onClick: handleFetchRefundableInfoClick,
        };
    }, [
        isCancelEnabled,
        refundableInfo.isError,
        renderIcon,
        handleFetchRefundableInfoClick,
    ]);

    useEffect(() => {
        if (cancelOrderInfo.isSuccess) {
            startOrderPolling({orderId});
        }
    }, [cancelOrderInfo.isSuccess, startOrderPolling, orderId]);

    useEffect(() => {
        fetchRefundableInfo(orderId);
    }, [orderId, fetchRefundableInfo]);

    const needsStandardActionButtons =
        STATUS_LIST_WITH_ACTION_BUTTONS.includes(status);

    if (inactive) {
        return null;
    }

    return (
        <Flex
            className={cx('root', deviceMods('root', deviceType), className)}
            flexDirection="row"
            justifyContent="space-between"
            alignItems={deviceType.isMobile ? 'initial' : 'center'}
            flexWrap="nowrap"
            inline
            between={2}
            {...prepareQaAttributes(props)}
        >
            <YandexMessengerButton
                entrypoint="hotelOrder"
                metrikaGoal={EHotelsGoal.ORDER_SUPPORT_BUTTON}
                targetPosition="bottom-right"
                buttonProps={{
                    icon: <MessageOutlineIcon className={cx('icon')} />,
                    children:
                        (deviceType.isDesktop || !needsStandardActionButtons) &&
                        i18nHappyPageBlock.supportButton(),
                    size: deviceType.isMobile ? 'l' : 'm-inset',
                    width:
                        deviceType.isMobile && !needsStandardActionButtons
                            ? 'max'
                            : undefined,
                }}
            />

            {needsStandardActionButtons && (
                <OrderActions
                    className={cx('actions')}
                    source={EOrderActionsSource.ORDER_PAGE}
                    print={printInfo}
                    download={downloadInfo}
                    cancel={cancelInfo}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'actions',
                    })}
                />
            )}

            <Modal
                isVisible={refundableInfo.isRefundModalVisible}
                isMobile={deviceType.isMobile}
                onClose={closeRefundModal}
                containerClassName={cx(
                    'modalContainer',
                    deviceModMobile('modalContainer', deviceType),
                )}
            >
                <div
                    className={cx(
                        'modalBox',
                        deviceMods('modalBox', deviceType),
                    )}
                >
                    <OrderHotelsCancelModal
                        orderId={orderId}
                        onClose={closeRefundModal}
                        startCancelOrder={handleCancelOrder}
                        orderCancelAndRefundableInfo={
                            orderCancelAndRefundableInfo
                        }
                    />
                </div>
            </Modal>
        </Flex>
    );
};

export default React.memo(OrderHotelsActions);
