import React from 'react';
import {useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';

import {ICancelOrderPayload} from 'reducers/hotels/bookAndPayPage/orderCancelAndRefundableInfo/actions';
import {IOrderCancelAndRefundableInfo} from 'reducers/hotels/bookAndPayPage/orderCancelAndRefundableInfo/reducer';

import hotelsBookInfoSelector from 'selectors/hotels/book/hotelsBookInfoSelector';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {deviceMods} from 'utilities/stylesUtils';
import getTotalPrice from 'projects/account/pages/Order/components/OrderHotels/utilities/getTotalPrice';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBlock from 'i18n/account-OrderHotels-Cancel';

import Button from 'components/Button/Button';
import Heading from 'components/Heading/Heading';
import Box from 'components/Box/Box';
import Spinner from 'components/Spinner/Spinner';
import Text from 'components/Text/Text';
import RefundDetails from 'projects/account/pages/Order/components/OrderHotels/components/RefundDetails/RefundDetails';

import cx from './OrderHotelsCancelModal.scss';

export interface ICancelOrderRefundableInfoProps extends IWithClassName {
    orderId: string;
    orderCancelAndRefundableInfo: IOrderCancelAndRefundableInfo;

    startCancelOrder(payload: ICancelOrderPayload): void;

    onClose?: () => void;
}

const CancelOrderRefundableInfo: React.FC<ICancelOrderRefundableInfoProps> = ({
    orderId,
    onClose,
    startCancelOrder,
    orderCancelAndRefundableInfo: {refundableInfo, cancelOrderInfo},
}) => {
    const {data: refundableInfoData} = refundableInfo;
    const deviceType = useDeviceType();

    const {orderInfo, offerInfoByToken} = useSelector(hotelsBookInfoSelector);

    const {isMobile, isDesktop} = deviceType;

    const handleStartCancelButtonClick = (
        e: React.MouseEvent<HTMLButtonElement>,
    ): void => {
        if (refundableInfoData) {
            const {refundToken} = refundableInfoData;

            startCancelOrder({refundToken, orderId});
        }

        e.preventDefault();
        e.stopPropagation();
    };

    const renderHotelName = (): React.ReactNode => {
        const hotelName = orderInfo.offerInfo?.hotelInfo.name;

        if (!hotelName) return;

        return (
            <Heading
                level={3}
                weight="medium"
                className={cx('hotelName', deviceMods('hotelName', deviceType))}
                size={isMobile ? 'm' : undefined}
            >
                {hotelName}
            </Heading>
        );
    };

    const renderConditionsInfo = (): React.ReactNode => {
        if (!refundableInfoData || !orderInfo) return;

        const priceInfo = offerInfoByToken?.offerInfo?.priceInfo;
        const orderPriceInfo = orderInfo.orderPriceInfo;
        const amountPaid = orderInfo.payment?.amountPaid;

        if (!amountPaid?.value) {
            return i18nBlock.freeCancellation();
        }

        const totalPrice = priceInfo
            ? getTotalPrice({
                  priceInfo,
                  orderPriceInfo,
              })
            : undefined;

        return (
            <RefundDetails
                totalPrice={totalPrice}
                amountPaid={amountPaid}
                refund={refundableInfoData.refund}
                penalty={refundableInfoData.penalty}
                appliedPromoCampaigns={orderInfo.appliedPromoCampaigns}
            />
        );
    };

    return (
        <section className={cx('content', deviceMods('content', deviceType))}>
            {(refundableInfo.isLoading ||
                cancelOrderInfo.isLoading ||
                cancelOrderInfo.isSuccess) && (
                <div className={cx('loader')}>
                    <Spinner />
                    {(cancelOrderInfo.isLoading ||
                        cancelOrderInfo.isSuccess) && (
                        <Text className={cx('loader_text')}>
                            {i18nBlock.cancelLoaderText()}
                        </Text>
                    )}
                </div>
            )}
            {refundableInfo.isError && (
                <div className={cx('error')}>{i18nBlock.errorDotFetch()}</div>
            )}
            {cancelOrderInfo.isError && (
                <div className={cx('error')}>{i18nBlock.errorDotCancel()}</div>
            )}
            {!refundableInfo.isError && !cancelOrderInfo.isError && (
                <>
                    <div
                        className={cx('block', deviceMods('block', deviceType))}
                    >
                        <Heading
                            level="2"
                            className={cx(
                                'title',
                                deviceMods('title', deviceType),
                            )}
                        >
                            {i18nBlock.orderCancelTitle()}
                        </Heading>
                        {renderHotelName()}
                        {renderConditionsInfo()}
                    </div>
                    <Box
                        className={cx('buttons')}
                        inline={isDesktop}
                        between="2"
                    >
                        <Button
                            onClick={handleStartCancelButtonClick}
                            size={isMobile ? 'l' : 'm-inset'}
                            theme="secondary"
                            width={isMobile ? 'max' : 'auto'}
                            {...prepareQaAttributes({
                                current: 'buttonSubmitCancelOrder',
                            })}
                        >
                            {i18nBlock.buttonDotSubmit()}
                        </Button>
                        <Button
                            onClick={onClose}
                            size={isMobile ? 'l' : 'm-inset'}
                            type="button"
                            theme="primary"
                            width={isMobile ? 'max' : 'auto'}
                        >
                            {i18nBlock.buttonDotCancel()}
                        </Button>
                    </Box>
                </>
            )}
        </section>
    );
};

export default CancelOrderRefundableInfo;
