import React, {useMemo} from 'react';

import {REFUNDED} from 'projects/hotels/constants/hotelsBookingStatuses';

import {IOrder} from 'server/api/HotelsBookAPI/types/IOrder';
import {IBookOffer} from 'server/api/HotelsBookAPI/types/IBookOffer';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {IWithClassName} from 'types/withClassName';

import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import getTotalPrice from 'projects/account/pages/Order/components/OrderHotels/utilities/getTotalPrice';

import * as i18nBlock from 'i18n/account-OrderHotels-CancelledInfo';
import * as i18nHappyPageBlock from 'i18n/happyPage';

import Heading from 'components/Heading/Heading';
import HotelsCancelledPenaltyInfo from 'projects/hotels/components/HotelsCancelledPenaltyInfo/HotelsCancelledPenaltyInfo';
import MessageOutlineIcon from 'icons/16/MessageOutline';
import YandexMessengerButton from 'components/YandexMessenger/components/YandexMessengerButton/YandexMessengerButton';
import Card from 'components/Card/Card';
import Separator from 'components/Separator/Separator';
import OrderHotelRefund from 'projects/account/pages/Order/components/OrderHotels/components/OrderHotelsCancelledInfo/components/OrderHotelRefund/OrderHotelRefund';

import cx from './OrderHotelsCancelledInfo.scss';

interface IOrderHotelsCancelledInfoProps extends IWithClassName {
    orderInfo: IOrder;
    offerInfo: IBookOffer;
}

const OrderHotelsCancelledInfo: React.FC<IOrderHotelsCancelledInfoProps> =
    props => {
        const {className, offerInfo, orderInfo} = props;
        const {priceInfo, cancellationInfo, hotelInfo} = offerInfo;
        const {
            orderPriceInfo,
            status,
            refundableInfo,
            guestsInfo,
            payment,
            appliedPromoCampaigns,
        } = orderInfo;
        const {refund, penalty, penaltyIntervalIndex} = refundableInfo;
        const {customerEmail} = guestsInfo;

        const deviceType = useDeviceType();

        const totalPrice = useMemo(() => {
            return getTotalPrice({
                priceInfo,
                orderPriceInfo,
            });
        }, [orderPriceInfo, priceInfo]);

        const penaltyInfoTitle = useMemo((): React.ReactNode => {
            const refundableRule =
                typeof penaltyIntervalIndex === 'number'
                    ? cancellationInfo?.refundableRules?.[penaltyIntervalIndex]
                    : undefined;

            return (
                <HotelsCancelledPenaltyInfo
                    className={cx('penaltyAmount')}
                    type={refundableRule?.type}
                    penalty={penalty}
                    startDate={refundableRule?.startDate}
                />
            );
        }, [cancellationInfo, penalty, penaltyIntervalIndex]);

        const supportButton = useMemo(() => {
            return (
                <YandexMessengerButton
                    className={cx('supportButton')}
                    entrypoint="hotelOrder"
                    metrikaGoal={EHotelsGoal.ORDER_SUPPORT_BUTTON}
                    targetPosition="bottom-right"
                    buttonProps={{
                        icon: <MessageOutlineIcon className={cx('icon')} />,
                        children: i18nHappyPageBlock.supportButton(),
                        size: deviceType.isMobile ? 'l' : 'm-inset',
                        width: deviceType.isMobile ? 'max' : undefined,
                    }}
                />
            );
        }, [deviceType.isMobile]);

        if (status !== REFUNDED) {
            return null;
        }

        return (
            <div
                className={cx(
                    className,
                    'root',
                    deviceMods('root', deviceType),
                )}
            >
                <Card
                    shadow={deviceType.isDesktop ? 'default' : undefined}
                    inset={deviceType.isDesktop ? 8 : undefined}
                >
                    <Heading
                        level="2"
                        size={deviceType.isMobile ? 'xl' : 'xxl'}
                        className={cx('title')}
                    >
                        {penalty && parseFloat(String(penalty.amount)) === 0
                            ? i18nBlock.titleDotFree()
                            : i18nBlock.titleDotPenalty()}
                    </Heading>
                    <div className={cx('text')}>
                        {insertJSXIntoKey(i18nBlock.mail)({
                            userEmail: (
                                <span className={cx('email')}>
                                    {customerEmail}
                                </span>
                            ),
                        })}
                    </div>
                    {penaltyInfoTitle && (
                        <div className={cx('text')}>{penaltyInfoTitle}</div>
                    )}
                    <Separator className={cx('refundSeparator')} />
                    <OrderHotelRefund
                        amountPaid={payment.amountPaid}
                        totalPrice={totalPrice}
                        refund={refund}
                        penalty={penalty}
                        appliedPromoCampaigns={appliedPromoCampaigns}
                        payment={payment}
                        hotelInfo={hotelInfo}
                    />
                    {deviceType.isMobile && supportButton}
                </Card>
                {deviceType.isDesktop && supportButton}
            </div>
        );
    };

export default React.memo(OrderHotelsCancelledInfo);
