import React, {useMemo} from 'react';

import {
    PAY_FISCAL_RECEIPT_TYPE,
    REFUND_FISCAL_RECEIPT_TYPE,
} from 'projects/hotels/constants/EFiscalReceiptType';

import {
    IOrderPaymentReceipt,
    IOrderPrice,
} from 'server/api/HotelsBookAPI/types/IOrder';
import IActionItem from 'components/ActionsDialog/types/IActionItem';
import {IBookHotelInfo} from 'server/api/HotelsBookAPI/types/IBookOffer';
import {IWithClassName} from 'types/withClassName';
import {IAppliedPromoCampaigns} from 'types/hotels/offer/IHotelOffer';

import IPrice from 'utilities/currency/PriceInterface';

import * as i18nBlock from 'i18n/account-OrderHotels-Price';

import LinkButton from 'components/LinkButton/LinkButton';
import useFiscalReceiptsActions from 'projects/account/pages/Order/components/OrderHotels/hooks/useFiscalReceiptsActions/useFiscalReceiptsActions';
import ActionsDialog from 'components/ActionsDialog/ActionsDialog';
import FiscalReceiptModal from 'projects/hotels/components/FiscalReceipt/components/FiscalReceiptModal/FiscalReceiptModal';
import RefundDetailsModal from 'projects/account/pages/Order/components/OrderHotels/components/OrderHotelsCancelledInfo/components/OrderHotelRefund/components/RefundReceiptsAndDocs/components/RefundDetailsModal/RefundDetailsModal';
import useDetailsActions from 'projects/account/pages/Order/components/OrderHotels/hooks/useDetailsActions';
import useActionsDialog from 'projects/account/pages/Order/components/OrderHotels/hooks/useActionsDialog';

import cx from './RefundReceiptsAndDocs.scss';

interface IRefundReceiptsAndDocsProps extends IWithClassName {
    amountPaid: IPrice | undefined;
    totalPrice: IPrice | undefined;
    refund: IOrderPrice | undefined;
    penalty: IOrderPrice | undefined;
    appliedPromoCampaigns: IAppliedPromoCampaigns | undefined;
    hotelInfo: IBookHotelInfo;
    receipts: IOrderPaymentReceipt[] | undefined;
}

const RefundReceiptsAndDocs: React.FC<IRefundReceiptsAndDocsProps> = props => {
    const {
        className,
        amountPaid,
        totalPrice,
        refund,
        penalty,
        appliedPromoCampaigns,
        hotelInfo,
        receipts,
    } = props;

    const actionsDialog = useActionsDialog();

    const refundDetailsAction = useDetailsActions({
        title: i18nBlock.refundDetails(),
    });
    const payFiscalReceiptsActions = useFiscalReceiptsActions({
        receipts,
        types: PAY_FISCAL_RECEIPT_TYPE,
        showPlaceholder: true,
    });
    const refundFiscalReceiptsActions = useFiscalReceiptsActions({
        receipts,
        types: REFUND_FISCAL_RECEIPT_TYPE,
        showPlaceholder: true,
    });

    const actions = useMemo((): IActionItem[] => {
        return [
            ...refundDetailsAction.actions,
            ...payFiscalReceiptsActions.actions,
            ...refundFiscalReceiptsActions.actions,
        ];
    }, [
        payFiscalReceiptsActions.actions,
        refundDetailsAction.actions,
        refundFiscalReceiptsActions.actions,
    ]);

    return (
        <div className={cx(className, 'root')}>
            <LinkButton
                className={cx('receiptsAndDocs')}
                innerRef={actionsDialog.ref}
                onClick={actionsDialog.open}
            >
                {i18nBlock.receiptsAndDocs()}
            </LinkButton>
            <ActionsDialog
                {...actionsDialog.commonProps}
                actions={actions}
                isOpened={actionsDialog.isOpened}
                onClose={actionsDialog.close}
            />
            <RefundDetailsModal
                amountPaid={amountPaid}
                totalPrice={totalPrice}
                refund={refund}
                penalty={penalty}
                appliedPromoCampaigns={appliedPromoCampaigns}
                hotelInfo={hotelInfo}
                isVisible={refundDetailsAction.isVisible}
                onClose={refundDetailsAction.close}
            />
            {[payFiscalReceiptsActions, refundFiscalReceiptsActions].map(
                (_actions, index) => {
                    return (
                        <FiscalReceiptModal
                            key={index}
                            backButtonText={hotelInfo?.name}
                            fiscalReceiptUrl={_actions.visibleReceiptsUrl}
                            isVisible={_actions.isVisible}
                            onClose={_actions.close}
                        />
                    );
                },
            )}
        </div>
    );
};

export default RefundReceiptsAndDocs;
