import {FunctionComponent} from 'react';

import {URLs} from 'constants/urls';

import {IWithClassName} from 'types/withClassName';
import {IOrderPayment} from 'server/api/HotelsBookAPI/types/IOrder';

import {internalUrl} from 'utilities/url';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {prepareDeferredPaymentDateTime} from 'projects/hotels/utilities/prepareDeferredPaymentDate/prepareDeferredPaymentDate';
import {HUMAN_DATETIME_WITHOUT_YEAR} from 'utilities/dateUtils/formats';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {CHAR_NBSP} from 'utilities/strings/charCodes';
import {deviceMods} from 'utilities/stylesUtils';
import IPrice from 'utilities/currency/PriceInterface';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/account-OrderHotelsDeferredPayment';

import ButtonLink from 'components/ButtonLink/ButtonLink';
import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Price from 'components/Price/Price';
import Separator from 'components/Separator/Separator';
import Card from 'components/Card/Card';
import PartialPrice from 'components/PartialPrice/PartialPrice';

import cx from './OrderHotelsDeferredPayment.scss';

export interface IOrderHotelsDeferredPaymentProps
    extends IWithClassName,
        IWithQaAttributes {
    orderId: string;
    payment: IOrderPayment;
    totalPrice?: IPrice;
}

const OrderHotelsDeferredPayment: FunctionComponent<IOrderHotelsDeferredPaymentProps> =
    props => {
        const {className, payment, orderId, totalPrice} = props;
        const {next, amountPaid} = payment;
        const deviceType = useDeviceType();
        const {isMobile, isDesktop} = deviceType;

        if (!next || !amountPaid || !totalPrice) {
            return null;
        }

        const {amount: nextPaymentPrice, paymentEndsAt} = next;

        return (
            <Card
                className={cx(
                    'root',
                    deviceMods('root', deviceType),
                    className,
                )}
                x={deviceType.isDesktop ? 8 : 3}
                y={deviceType.isDesktop ? 8 : 5}
                shadow={deviceType.isMobile ? undefined : 'default'}
                border={deviceType.isMobile ? 'thin' : undefined}
                {...prepareQaAttributes(props)}
            >
                <Flex flexDirection="column">
                    <Flex
                        className={cx('paid')}
                        justifyContent="space-between"
                        alignItems="center"
                    >
                        <Text size="m" color="primary">
                            {i18nBlock.amountPaidLabel()}
                        </Text>
                        <Text size="l" color="primary">
                            <PartialPrice
                                price={amountPaid}
                                totalPrice={totalPrice}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'partialPrice',
                                })}
                            />
                        </Text>
                    </Flex>
                    <Separator />
                    <Flex
                        className={cx('nextPayment')}
                        alignItems="center"
                        flexDirection={isDesktop ? 'row' : 'column-reverse'}
                    >
                        <Text size="m" color="alert">
                            {insertJSXIntoKey(i18nBlock.deferredPaymentInfo)({
                                dateNode: (
                                    <span
                                        className={cx('dateLabel')}
                                        {...prepareQaAttributes({
                                            parent: props,
                                            current: 'paymentEndsAt',
                                        })}
                                    >
                                        {prepareDeferredPaymentDateTime(
                                            paymentEndsAt,
                                            HUMAN_DATETIME_WITHOUT_YEAR,
                                        )}
                                    </span>
                                ),
                            })}
                        </Text>
                        {isDesktop && (
                            <Text
                                className={cx('nextPaymentPrice')}
                                size="l"
                                color="alert"
                            >
                                <Price
                                    value={nextPaymentPrice.value}
                                    currency={nextPaymentPrice.currency}
                                    {...prepareQaAttributes({
                                        parent: props,
                                        current: 'nextPaymentPrice',
                                    })}
                                />
                            </Text>
                        )}
                        <ButtonLink
                            theme="primary"
                            size={isDesktop ? 'm-inset' : 'l'}
                            url={internalUrl(
                                URLs.hotelsBookStartDeferredPayment,
                                {
                                    orderId,
                                },
                            )}
                            target="_blank"
                            width={isDesktop ? 'auto' : 'max'}
                            className={cx('buttonLink')}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'nextPaymentLink',
                            })}
                        >
                            {i18nBlock.startDeferredPaymentLink()}
                            {isMobile && (
                                <>
                                    {CHAR_NBSP}
                                    <Price
                                        value={nextPaymentPrice.value}
                                        currency={nextPaymentPrice.currency}
                                        {...prepareQaAttributes({
                                            parent: props,
                                            current: 'nextPaymentPrice',
                                        })}
                                    />
                                </>
                            )}
                        </ButtonLink>
                    </Flex>
                </Flex>
            </Card>
        );
    };

export default OrderHotelsDeferredPayment;
