import React, {useMemo} from 'react';

import {
    REFUND_FAILED,
    CANCELLED_WITH_REFUND,
    FAILED,
} from 'projects/hotels/constants/hotelsBookingStatuses';

import {IWithClassName} from 'types/withClassName';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/account-OrderHotels-Error';

import CardWithDeviceLayout from 'components/CardWithDeviceLayout/CardWithDeviceLayout';
import BaseError from './components/BaseError/BaseError';

import cx from './OrderHotelsError.scss';

export interface IOrderHotelsErrorProps
    extends IWithClassName,
        IWithQaAttributes {
    status: string;
    yandexOrderId: string;
}

const OrderHotelsError: React.FC<IOrderHotelsErrorProps> = props => {
    const {status, yandexOrderId, className} = props;
    const rootQa = getQa(props);

    const deviceType = useDeviceType();

    const content = useMemo(() => {
        switch (status) {
            case REFUND_FAILED:
                return (
                    <BaseError
                        yandexOrderId={yandexOrderId}
                        title={i18nBlock.refundDotTitle()}
                        description={i18nBlock.refundDotInfo()}
                        action={i18nBlock.refundDotAction()}
                        supportContactTitle={i18nBlock.refundDotSupport()}
                        {...prepareQaAttributes(rootQa)}
                    />
                );
            case FAILED:
                return (
                    <BaseError
                        yandexOrderId={yandexOrderId}
                        title={i18nBlock.failedDotInfo()}
                        description={i18nBlock.failedDotInfo()}
                        {...prepareQaAttributes(rootQa)}
                    />
                );
            case CANCELLED_WITH_REFUND:
                return (
                    <BaseError
                        yandexOrderId={yandexOrderId}
                        title={i18nBlock.cancelledDotTitle()}
                        description={i18nBlock.cancelledDotInfo()}
                        {...prepareQaAttributes(rootQa)}
                    />
                );
            default:
                return null;
        }
    }, [rootQa, status, yandexOrderId]);

    if (!content) {
        return null;
    }

    return (
        <CardWithDeviceLayout
            className={cx(className, 'root', deviceMods('root', deviceType))}
            {...prepareQaAttributes(rootQa)}
        >
            {content}
        </CardWithDeviceLayout>
    );
};

export default React.memo(OrderHotelsError);
