import React, {useMemo} from 'react';

import {
    TBookOfferBedsGroups,
    IBookHotelSettlement,
    IBookOfferMeal,
    IBookOfferRoom,
} from 'server/api/HotelsBookAPI/types/IBookOffer';
import {IWithClassName} from 'types/withClassName';
import {isNotUndefined} from 'types/utilities';

import {deviceMods} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import Text from 'components/Text/Text';
import BedsGroups from 'projects/hotels/components/BedsGroups/BedsGroups';
import BookSettlementInfo from 'projects/hotels/components/BookSettlementInfo/BookSettlementInfo';
import BookHotelAmenities from 'projects/hotels/components/BookHotelAmenities/BookHotelAmenities';
import BookMealInfo from 'projects/hotels/components/BookMealInfo/BookMealInfo';
import Card from 'components/Card/Card';
import ImagesSwipeCarousel from 'components/ImagesSwipeCarousel/ImagesSwipeCarousel';
import Separator from 'components/Separator/Separator';
import ImagesCarousel from 'components/ImagesCarousel/ImagesCarousel';

import cx from './OrderHotelsInfo.scss';

interface IOrderHotelsInfoProps extends IWithClassName, IWithQaAttributes {
    settlementInfo: IBookHotelSettlement;
    roomInfo: IBookOfferRoom;
    bedsGroups: TBookOfferBedsGroups;
    mealInfo: IBookOfferMeal;
    inactive: boolean;
}

const OrderHotelsInfo: React.FC<IOrderHotelsInfoProps> = props => {
    const {
        className,
        roomInfo,
        bedsGroups,
        mealInfo,
        settlementInfo,
        inactive,
    } = props;

    const deviceType = useDeviceType();

    const imagesNode = useMemo(() => {
        const images = roomInfo.images
            ?.map(img => ({
                src: img?.links?.['350px']?.href,
            }))
            .filter(isNotUndefined);

        if (!images?.length) {
            return null;
        }

        if (deviceType.isMobile) {
            return (
                <ImagesSwipeCarousel
                    borderRadius="l"
                    images={images}
                    swipeableClassName={cx('imagesCarouselContent')}
                />
            );
        }

        return (
            <ImagesCarousel
                images={images}
                arrowsType="dark"
                type="mini"
                isDesktop={deviceType.isDesktop}
            />
        );
    }, [deviceType, roomInfo]);

    return (
        <Card
            className={cx(className, 'root', deviceMods('root', deviceType), {
                root_inactive: inactive,
            })}
            shadow={deviceType.isDesktop ? 'default' : undefined}
            inset={deviceType.isDesktop ? 8 : undefined}
            {...prepareQaAttributes(props)}
        >
            {imagesNode && (
                <div
                    className={cx('images')}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'images',
                    })}
                >
                    {imagesNode}
                </div>
            )}
            <div className={cx('roomInfo')}>
                <Text
                    className={cx('roomName')}
                    size="xl"
                    weight="medium"
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'roomName',
                    })}
                >
                    {roomInfo.name}
                </Text>
                <BedsGroups
                    className={cx('bedsGroups')}
                    bedsGroups={bedsGroups}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'bedGroups',
                    })}
                />
                {deviceType.isDesktop && <Separator margin={8} />}
                <BookHotelAmenities
                    className={cx('amenities')}
                    amenities={roomInfo.amenities}
                />
                <BookMealInfo
                    className={cx('mealInfo')}
                    mealInfo={mealInfo}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'mealInfo',
                    })}
                />
                <BookSettlementInfo
                    className={cx('settlementInfo')}
                    settlementInfo={settlementInfo}
                    isMobile={deviceType.isMobile}
                />
            </div>
        </Card>
    );
};

export default React.memo(OrderHotelsInfo);
