import React, {useCallback, useMemo, useState} from 'react';
import moment, {Moment} from 'moment';

import {IOrderRefundableInfo} from 'server/api/HotelsBookAPI/types/IOrder';
import {IBookOfferRefundableRuleAll} from 'projects/account/pages/Order/types/OrderHotelsCancellationProps';
import EBookOfferRefundableType from 'projects/account/pages/Order/types/EBookOfferRefundableType';
import {IBookOfferCancellationInfo} from 'server/api/HotelsBookAPI/types/IBookOffer';
import {IWithClassName} from 'types/withClassName';

import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {formatDate, getNow} from 'utilities/dateUtils';
import {HUMAN, TIME} from 'utilities/dateUtils/formats';
import {deviceMods} from 'utilities/stylesUtils';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import getHeadingText from './utilities/getHeadingText';
import getWarning from './utilities/getWarning';
import getPenaltyText from 'projects/account/pages/Order/components/OrderHotels/components/OrderHotelsRefundInfo/utilities/getPenaltyText';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import getCancellationInfoAll from 'projects/account/pages/Order/components/OrderHotels/components/OrderHotelsRefundInfo/utilities/getCancellationInfoAll';

import * as i18nBlock from 'i18n/account-OrderHotels-RefundInfo';

import NotificationBanner from 'components/NotificationBanner/NotificationBanner';
import Text from 'components/Text/Text';
import Card from 'components/Card/Card';

import cx from './OrderHotelsRefundInfo.scss';

export interface IOrderHotelsRefundInfoProps extends IWithClassName {
    cancellationInfoAll: IBookOfferCancellationInfo;
    refundableInfo: IOrderRefundableInfo;
}

const OrderHotelsRefundInfo: React.FC<IOrderHotelsRefundInfoProps> = props => {
    const {className, refundableInfo} = props;

    const deviceType = useDeviceType();

    const [elementPosition, setElementPosition] = useState({
        left: 0,
        right: 0,
    });
    const [containerPosition, setContainerPosition] = useState({
        left: 0,
        right: 0,
    });

    const cancellationInfoAll = useMemo(
        () =>
            getCancellationInfoAll({
                cancellationInfoAll: props.cancellationInfoAll,
                refundableInfo,
            }),
        [props.cancellationInfoAll, refundableInfo],
    );

    const handleTodayGetLeftOffset = useCallback(
        (ref: HTMLDivElement): void => {
            if (ref) {
                const {left, width} = ref.getBoundingClientRect();

                setElementPosition({
                    left,
                    right: left + width,
                });
            }
        },
        [],
    );
    const setContainerBounds = useCallback((ref: HTMLDivElement): void => {
        if (ref) {
            const {left, width} = ref.getBoundingClientRect();

            setContainerPosition({
                left,
                right: left + width,
            });
        }
    }, []);

    const renderCurrentLine = useCallback(
        (startDate: Moment, endDate: Moment, current: boolean) => {
            const {refundedDate} = cancellationInfoAll;

            const thisDate = refundedDate || getNow();

            const allDiff = moment(endDate).diff(startDate, 'h');
            const nowDiff = moment(thisDate).diff(startDate, 'h');

            const conditionLeft = containerPosition.left > elementPosition.left;
            const conditionRight =
                containerPosition.right < elementPosition.right;

            return (
                <div className={cx('currentBlock')}>
                    {current && (
                        <div
                            className={cx('current')}
                            style={{
                                paddingLeft: `${(nowDiff / allDiff) * 100}%`,
                            }}
                        >
                            <div className={cx('today-pipe')} />
                            <div
                                className={cx('today', {
                                    today_left: conditionLeft,
                                    today_normal:
                                        !conditionLeft && !conditionRight,
                                    today_right: conditionRight,
                                })}
                                ref={handleTodayGetLeftOffset}
                            >
                                {(refundedDate
                                    ? i18nBlock.refundedDate
                                    : i18nBlock.today)({
                                    date: formatDate(thisDate, HUMAN),
                                })}
                            </div>
                        </div>
                    )}
                </div>
            );
        },
        [
            cancellationInfoAll,
            containerPosition,
            elementPosition,
            handleTodayGetLeftOffset,
        ],
    );

    const renderSingleLine = useCallback(
        (refundableRule: IBookOfferRefundableRuleAll, index: number) => {
            const {type, startDate, endDate, past, current} = refundableRule;

            const text = getPenaltyText(refundableRule, deviceType);

            if (!text) {
                return null;
            }

            const date = formatDate(startDate, HUMAN);
            const time = formatDate(startDate, TIME);

            return (
                <div
                    key={index}
                    className={cx('item', {
                        item_free: type === EBookOfferRefundableType.NO_PENALTY,
                        item_some:
                            type === EBookOfferRefundableType.SOME_PENALTY,
                        item_full: type === EBookOfferRefundableType.FULL_PRICE,
                        item_past: past,
                    })}
                >
                    <div className={cx('top')}>
                        <div className={cx('date')}>
                            {insertJSXIntoKey(i18nBlock.date)({
                                date,
                                time: type !==
                                    EBookOfferRefundableType.NO_PENALTY && (
                                    <span className={cx('time')}>{time}</span>
                                ),
                            })}
                        </div>
                        <div className={cx('text')}>{text}</div>
                    </div>
                    <div className={cx('line')} />
                    {renderCurrentLine(startDate, endDate, current)}
                </div>
            );
        },
        [deviceType, renderCurrentLine],
    );

    const {refundable, hasTimeline, refundableRules} = cancellationInfoAll;

    if ((!refundable || hasTimeline) && !refundableInfo.penaltyIntervalIndex) {
        return (
            <NotificationBanner
                className={className}
                {...prepareQaAttributes('orderWarning')}
            >
                <Text size="m">{i18nBlock.warningDotNoCancel()}</Text>
            </NotificationBanner>
        );
    }

    const title = getHeadingText(cancellationInfoAll.currentPenalty);

    return (
        <Card
            className={cx(className, 'root', deviceMods('root', deviceType))}
            x={deviceType.isMobile ? 4 : 8}
            y={deviceType.isMobile ? 4 : 5}
            radius={deviceType.isMobile ? 's' : 'm'}
            background="grey"
        >
            {title && (
                <div className={cx('name')}>
                    {title}
                    {cancellationInfoAll.highlighted &&
                        getWarning(cancellationInfoAll)}
                </div>
            )}
            <div className={cx('items')} ref={setContainerBounds}>
                {refundableRules.map(renderSingleLine)}
            </div>
        </Card>
    );
};

export default React.memo(OrderHotelsRefundInfo);
