import moment, {Moment} from 'moment-timezone';
import _isEmpty from 'lodash/isEmpty';

import {IBookOfferCancellationInfo} from 'server/api/HotelsBookAPI/types/IBookOffer';
import {IOrderRefundableInfo} from 'server/api/HotelsBookAPI/types/IOrder';
import {IBookOfferCancellationInfoAll} from 'projects/account/pages/Order/types/OrderHotelsCancellationProps';
import EBookOfferRefundableType, {
    isBookOfferRefundableType,
} from 'projects/account/pages/Order/types/EBookOfferRefundableType';

import {MOSCOW_TIME_ZONE} from 'utilities/dateUtils/timezones';
import {DateLikeType, getNow} from 'utilities/dateUtils';

interface IGetCancellationInfoAllParams {
    cancellationInfoAll: IBookOfferCancellationInfo | undefined;
    refundableInfo: IOrderRefundableInfo;
}

export default function getCancellationInfoAll({
    cancellationInfoAll,
    refundableInfo,
}: IGetCancellationInfoAllParams): IBookOfferCancellationInfoAll {
    if (!cancellationInfoAll) {
        return {
            refundable: false,
            highlighted: false,
            hasTimeline: false,
            refundableRules: [],
        };
    }

    const {refundable, refundableRules, highlighted} = cancellationInfoAll;

    let highlightDaysLeft, highlightDate, highlightDateSame, highlightNextType;
    let currentPenalty, refundedDate;

    const refundableRulesNew = refundableRules.map((penality, index) => {
        const {
            type: typeString,
            price,
            endDate: endDateString,
            startDate: startDateString,
        } = penality;

        const startDate = prepareRefundableDate(startDateString);
        const endDate = prepareRefundableDate(endDateString);
        let nowDate;

        const type = isBookOfferRefundableType(typeString)
            ? typeString
            : EBookOfferRefundableType.UNKNOWN;

        let current = false,
            past = false;

        if (_isEmpty(refundableInfo)) {
            nowDate = prepareRefundableDate(getNow());

            current =
                (startDate ? nowDate.isAfter(startDate) : true) &&
                (endDate ? nowDate.isBefore(endDate) : true);
            past = Boolean(endDate && nowDate.isAfter(endDate));

            if (current) {
                currentPenalty = {
                    type,
                    price,
                    startDate,
                    endDate,
                };

                if (highlighted && refundableRules[index + 1]) {
                    highlightDaysLeft = endDate.diff(nowDate, 'days', true);
                    highlightDate = endDate;

                    if (highlightDaysLeft < 1) {
                        highlightDaysLeft = 0;

                        highlightDateSame = moment(endDate).isSame(
                            nowDate,
                            'days',
                        );
                    }

                    highlightDaysLeft = Math.round(highlightDaysLeft);

                    const nextPenalty = refundableRules[index + 1];

                    if (
                        type === EBookOfferRefundableType.SOME_PENALTY &&
                        nextPenalty.type ===
                            EBookOfferRefundableType.SOME_PENALTY
                    ) {
                        highlightNextType =
                            EBookOfferRefundableType.MORE_PENALTY;
                    } else {
                        highlightNextType = nextPenalty.type;
                    }
                }
            }
        } else if (
            refundableInfo.penaltyIntervalIndex !== undefined &&
            refundableInfo.refundDateTime
        ) {
            current = index === refundableInfo.penaltyIntervalIndex;
            past = index < refundableInfo.penaltyIntervalIndex;
            refundedDate = prepareRefundableDate(refundableInfo.refundDateTime);

            if (current) {
                currentPenalty = {
                    type,
                    price,
                    startDate,
                    endDate,
                };
            }
        }

        return {
            price,
            type,
            endDate,
            startDate,
            current,
            past,
        };
    });

    return {
        refundable,
        highlighted,
        refundableRules: refundableRulesNew,
        currentPenalty,
        highlightDaysLeft,
        highlightDate,
        highlightNextType,
        highlightDateSame,
        refundedDate,
        hasTimeline: refundableRulesNew.every(item => item.past),
    };
}

function prepareRefundableDate(date: DateLikeType): Moment {
    return moment.utc(date).tz(MOSCOW_TIME_ZONE);
}
