import React from 'react';

import {IWithClassName} from 'types/withClassName';
import {
    EYandexPlusApplicationMode,
    IAppliedPromoCampaigns,
} from 'types/hotels/offer/IHotelOffer';
import {IOrderPrice} from 'server/api/HotelsBookAPI/types/IOrder';
import {offerPriceToPrice} from 'types/common/price/offerPriceToPrice';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {IWithQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import IPrice from 'utilities/currency/PriceInterface';

import * as i18nBlock from 'i18n/account-OrderHotels-Price';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Price from 'components/Price/Price';
import Separator from 'components/Separator/Separator';
import PartialPrice from 'components/PartialPrice/PartialPrice';
import DetailItem from 'projects/account/pages/Order/components/OrderHotels/components/DetailItem/DetailItem';
import YandexPlusSignBwS from 'icons/illustrations/YandexPlusSignBw/YandexPlusSignBwS';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import StriketroughText from 'components/StriketroughText/StriketroughText';
import PlusRefundDescription from 'projects/account/pages/Order/components/OrderHotels/components/RefundDetails/components/PlusRefundDescription/PlusRefundDescription';

import cx from './RefundDetails.scss';

interface IDetailsProps extends IWithClassName, IWithQaAttributes {
    totalPrice: IPrice | undefined;
    amountPaid: IPrice | undefined;
    refund: IOrderPrice | undefined;
    penalty: IOrderPrice | undefined;
    appliedPromoCampaigns: IAppliedPromoCampaigns | undefined;
}

const RefundDetails: React.FC<IDetailsProps> = props => {
    const {
        className,
        amountPaid,
        totalPrice,
        refund,
        penalty,
        appliedPromoCampaigns,
    } = props;

    const deviceType = useDeviceType();

    const penaltyAsPrice = penalty ? offerPriceToPrice(penalty) : undefined;

    const yandexPlus = appliedPromoCampaigns?.yandexPlus;

    return (
        <div className={cx(className, 'root', deviceMods('root', deviceType))}>
            <Flex flexDirection="column" between={3}>
                {totalPrice && (
                    <DetailItem
                        title={i18nBlock.paidTitle()}
                        price={
                            <PartialPrice
                                price={amountPaid}
                                totalPrice={totalPrice}
                            />
                        }
                    />
                )}
                {penaltyAsPrice && Boolean(penaltyAsPrice.value) && (
                    <DetailItem
                        title={i18nBlock.penaltyTitle()}
                        price={<Price {...penaltyAsPrice} />}
                    />
                )}
                {yandexPlus &&
                    yandexPlus.mode === EYandexPlusApplicationMode.TOPUP &&
                    yandexPlus.points > 0 && (
                        <DetailItem
                            title={i18nBlock.refundNoPlusCashback()}
                            price={
                                <TextWithIcon
                                    text={
                                        <StriketroughText
                                            crossed
                                            size="inherit"
                                        >
                                            {yandexPlus.points}
                                        </StriketroughText>
                                    }
                                    size="m"
                                    spaceBetween="1"
                                    iconLeftClassName={cx('plusIcon')}
                                    iconLeft={YandexPlusSignBwS}
                                    iconSize={16}
                                />
                            }
                        />
                    )}
            </Flex>

            {refund && (
                <>
                    <Separator margin={4} />
                    <Flex flexDirection="column" between={3}>
                        <DetailItem
                            title={
                                <Text size="inherit" weight="bold">
                                    {i18nBlock.refundTitle()}
                                </Text>
                            }
                            price={
                                <Text size="inherit" weight="bold">
                                    <Price
                                        value={refund.amount}
                                        currency={refund.currency}
                                    />
                                </Text>
                            }
                        />
                        <PlusRefundDescription yandexPlus={yandexPlus} />
                    </Flex>
                </>
            )}
        </div>
    );
};

export default RefundDetails;
