import {useCallback, useMemo, useState} from 'react';

import {EFiscalReceiptType} from 'projects/hotels/constants/EFiscalReceiptType';

import {IOrderPaymentReceipt} from 'server/api/HotelsBookAPI/types/IOrder';
import IActionItem from 'components/ActionsDialog/types/IActionItem';

import getFiscalReceiptText from './utilities/getFiscalReceiptText';

import * as i18nBlock from 'i18n/hotels-FiscalReceipt';

import CheckIcon from 'icons/24/Check';
import Text from 'components/Text/Text';

interface IParams {
    receipts: IOrderPaymentReceipt[] | undefined;
    types: EFiscalReceiptType[];
    showPlaceholder: boolean;
}

interface IResult {
    actions: IActionItem[];
    visibleReceiptsUrl: string;
    isVisible: boolean;
    close(): void;
}

export default function useFiscalReceiptsActions({
    receipts,
    types,
    showPlaceholder,
}: IParams): IResult {
    const [receiptState, setReceiptState] = useState({
        visibleReceiptsUrl: '',
        isVisible: false,
    });

    const handleCloseReceipt = useCallback(() => {
        setReceiptState(state => ({...state, isVisible: false}));
    }, []);

    const actions = useMemo(() => {
        const fiscalReceipts =
            receipts?.filter(({type}) => types.includes(type)) ?? [];

        if (showPlaceholder && fiscalReceipts.length === 0) {
            const placeholderAction: IActionItem = {
                id: `receipt-${types.join('-')}-empty`,
                title: getFiscalReceiptText({
                    type: types[0],
                    isSingle: true,
                    index: 0,
                }),
                disabled: true,
                description: (
                    <Text size="inherit">
                        {i18nBlock.emptyFiscalReceiptTitle()}
                    </Text>
                ),
                icon: CheckIcon,
                onClick: (): void => {},
            };

            return [placeholderAction];
        }

        return fiscalReceipts.map((receipt, index) => {
            const isSingle = fiscalReceipts.length === 1;

            return {
                id: `receipt-${receipt.type}-${index}`,
                title: getFiscalReceiptText({
                    type: receipt.type,
                    isSingle,
                    index,
                }),
                icon: CheckIcon,
                onClick: (): void => {
                    setReceiptState({
                        isVisible: true,
                        visibleReceiptsUrl: receipt.url,
                    });
                },
            };
        });
    }, [receipts, showPlaceholder, types]);

    return {
        actions,
        visibleReceiptsUrl: receiptState.visibleReceiptsUrl,
        isVisible: receiptState.isVisible,
        close: handleCloseReceipt,
    };
}
