import {useCallback, useEffect, useMemo, useRef} from 'react';

import {SECOND} from 'utilities/dateUtils/constants';

import IActionItem from 'components/ActionsDialog/types/IActionItem';

import {delay} from 'utilities/async/delay';
import {useAsyncState} from 'utilities/hooks/useAsyncState';
import {logError} from 'utilities/logger/logError';
import {getHotelsBusinessTripPdfUrl} from 'projects/hotels/utilities/getHotelsBusinessTripPdfUrl/getHotelsBusinessTripPdfUrl';

import * as i18nBlock from 'i18n/account-OrderHotels-Price';

import DownloadIcon from 'icons/24/Download';
import Text from 'components/Text/Text';

import {browserHotelBookProvider} from 'serviceProvider/hotels/book/browserHotelBookProvider';

const MAX_TRIES = 10;

interface IParams {
    canGenerateBusinessTripDoc: boolean | undefined;
    orderId: string;
    inactive: boolean;
}

interface IResult {
    actions: IActionItem[];
}

export default function useGenerateBusinessTripPdfActions({
    canGenerateBusinessTripDoc,
    orderId,
    inactive,
}: IParams): IResult {
    const abortRef = useRef(false);
    const asyncState = useAsyncState();

    const description = useMemo(() => {
        if (!canGenerateBusinessTripDoc) {
            return (
                <Text size="inherit" color="alert">
                    {i18nBlock.businessTripDocDisableDescription()}
                </Text>
            );
        }

        if (asyncState.isError) {
            return (
                <Text size="inherit" color="alert">
                    {i18nBlock.businessTripDocError()}
                </Text>
            );
        }

        return null;
    }, [asyncState.isError, canGenerateBusinessTripDoc]);

    useEffect(() => {
        return () => {
            abortRef.current = true;
        };
    }, []);

    const handleClick = useCallback(async (): Promise<void> => {
        try {
            asyncState.loading();

            await browserHotelBookProvider.generateBusinessTripPdf({
                id: orderId,
            });

            for (let i = 0; i < MAX_TRIES; i++) {
                const {isReady} =
                    await browserHotelBookProvider.checkBusinessTripPdf({
                        id: orderId,
                    });

                if (abortRef.current) {
                    asyncState.reset();

                    return;
                }

                if (isReady) {
                    asyncState.success();

                    window.open(getHotelsBusinessTripPdfUrl(orderId), '_self');

                    return;
                }

                await delay(5 * SECOND);
            }

            throw new Error(
                `Business trip PDF not ready after ${MAX_TRIES} tries`,
            );
        } catch (e) {
            asyncState.error();

            logError(
                {
                    message:
                        'useGenerateBusinessTripPdfActions: fail generate business trip PDF',
                },
                e,
            );

            throw e;
        }
    }, [asyncState, orderId]);

    const actions = useMemo(() => {
        if (canGenerateBusinessTripDoc === undefined || inactive) {
            return [];
        }

        const action = {
            id: 'businessTripDoc',
            title: i18nBlock.businessTripDoc(),
            icon: DownloadIcon,
            loading: asyncState.isLoading,
            disabled: !canGenerateBusinessTripDoc || asyncState.isLoading,
            description,
            onClick: handleClick,
        };

        return [action];
    }, [
        asyncState.isLoading,
        canGenerateBusinessTripDoc,
        description,
        handleClick,
        inactive,
    ]);

    return {actions};
}
