import React, {useEffect, useRef} from 'react';

import {IWithClassName} from 'types/withClassName';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {useBoolean} from 'utilities/hooks/useBoolean';

import Box from 'components/Box/Box';
import MessageBoxPopup from 'components/MessageBoxPopup/MessageBoxPopup';

const TOOLTIP_DIRECTIONS = [EPopupDirection.BOTTOM];

interface ITooltipProps extends IWithClassName {
    /**
     * Если указано false, то тултип закроется и перестанет реагировать на действия пользователя
     */
    active: boolean;

    /**
     * Контент для отрисовки в тултипе
     */
    content: React.ReactNode;

    /**
     * Направления для раскрытия подсказки
     */
    directions?: EPopupDirection[];
}

/**
 * Тултип, раскрывающийся при клике по дочернему элементу
 */
const Tooltip: React.FC<ITooltipProps> = props => {
    const {
        className,
        active,
        content,
        directions = TOOLTIP_DIRECTIONS,
        children,
    } = props;
    const nodeRef = useRef<HTMLDivElement>(null);
    const {
        value: isVisible,
        toggle: toggleVisibility,
        setFalse: hideTooltip,
    } = useBoolean(false);

    useEffect(() => {
        if (!active) {
            hideTooltip();
        }
    }, [active, hideTooltip]);

    return (
        <div
            className={className}
            onClick={active ? toggleVisibility : undefined}
            ref={nodeRef}
        >
            {children}
            <MessageBoxPopup
                isVisible={isVisible}
                anchorRef={nodeRef}
                direction={directions}
                onClose={hideTooltip}
            >
                <Box x={3} y={2}>
                    {content}
                </Box>
            </MessageBoxPopup>
        </div>
    );
};

export default Tooltip;
