import {PureComponent, ReactNode} from 'react';
import {connect} from 'react-redux';
import {bindActionCreators, Dispatch} from 'redux';
import {withRouter} from 'react-router-dom';
import Helmet from 'react-helmet';
import _partial from 'lodash/partial';
import {RouteComponentProps} from 'react-router';

import {URLs} from 'constants/urls';

import {IWithDeviceType} from 'types/withDeviceType';
import {ICountry} from 'types/common/ICountry';
import {IPassengerDocumentData} from 'types/common/document/IPassengerDocumentData';

import * as passengerActions from 'reducers/account/passengers/actions';
import * as docActions from 'reducers/account/docs/actions';
import {requestCountries as requestCountriesAction} from 'reducers/common/countries/actions';
import {StoreInterface} from 'reducers/storeTypes';
import {IPreparedPassenger} from 'reducers/account/passengers/api-types';

import getDocuments from 'selectors/account/documentsSelector';
import getPassengers from 'selectors/account/passengerSelector';
import getDocTypes from 'selectors/account/docTypesSelector';
import {getUserInfo} from 'selectors/common/userInfoSelector';
import {
    getAllCountriesData,
    getAllCountries,
    getCountriesReducerState,
} from 'selectors/common/countriesSelectors';

import {internalUrl} from 'utilities/url';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {getPassportUrl, EPassportMode} from 'utilities/url/getPassportUrl';
import {accountURLs} from 'projects/account/utilities/urls';
import {reloadPage} from 'utilities/browser/reloadPage';
import TDataAddPassenger from 'utilities/passengerApiMethods/TDataAddPassenger';
import goBack from 'utilities/browser/goBack';

import * as i18nCommonBlock from 'i18n/common';
import * as i18nBlock from 'i18n/account';

import Select from 'components/Select/Select';
import PassengerList from './components/PassengerList/PassengerList';
import PassengerInfo from './components/PassengerInfo/PassengerInfo';
import PassengerRemove from './components/PassengerRemove/PassengerRemove';
import PassengerDocs from './components/Documents/Documents';
import Preloader from 'projects/account/components/AccountPreloader/AccountPreloader';
import AccountModal from 'projects/account/components/AccountModal/AccountModal';
import AccountError from 'projects/account/components/AccountError/AccountError';

import cx from './AccountPassengerPage.scss';

const TOP_COUNTRIES_COUNT = 4;
const ROOT_QA = 'passenger-page';

interface IOwnProps
    extends RouteComponentProps<{userId: string}>,
        IWithDeviceType {}

const mapStateToProps = (state: StoreInterface, props: IOwnProps) => {
    const passengers = getPassengers(state);

    let {userId} = props.match.params;

    if (!userId && passengers.items.length && !props.deviceType.isMobile) {
        userId = passengers.items[0].id;
    }

    const countriesState = getCountriesReducerState(state);

    const allCountries: (ICountry | typeof Select.Separator)[] =
        getAllCountries(state).slice();

    allCountries.splice(TOP_COUNTRIES_COUNT, 0, Select.Separator);

    return {
        ...passengers,
        fetched: passengers.fetched && countriesState.isFetched,
        fetching: passengers.fetching || countriesState.isLoading,
        error: passengers.error || countriesState.isError,
        authError: passengers.authError,
        documents: getDocuments(state, {userId}),
        docTypes: getDocTypes(state),
        citizenshipData: getAllCountriesData(state),
        citizenshipForSelect: allCountries,
        userId: userId,
        userInfo: getUserInfo(state),
    };
};

const mapDispatchToProps = (dispatch: Dispatch) =>
    bindActionCreators(
        {
            ...passengerActions,
            ...docActions,
            requestCountries: requestCountriesAction,
        },
        dispatch,
    );

interface IAccountPassengerPageProps
    extends IOwnProps,
        ReturnType<typeof mapStateToProps>,
        ReturnType<typeof mapDispatchToProps> {}

interface IAccountPassengerPageState {
    isModalOpened: boolean;
}

class AccountPassengerPage extends PureComponent<
    IAccountPassengerPageProps,
    IAccountPassengerPageState
> {
    state = {
        isModalOpened: false,
    };

    componentDidMount(): void {
        const {fetched, fetching, fetchPassenger, requestCountries} =
            this.props;

        if (!fetched && !fetching) {
            fetchPassenger();
            requestCountries();
        }
    }

    componentDidUpdate(): void {
        const {
            fetched,
            fetching,
            authError,
            userId,
            items,
            lastInsertedId,
            deviceType: {isMobile, isDesktop},
        } = this.props;
        const {isModalOpened} = this.state;

        if (fetched && !fetching) {
            if (userId) {
                const passenger = items.find(
                    (item: {id: string}) => item.id === userId,
                );

                if (!passenger || (isDesktop && lastInsertedId)) {
                    this.redirect();
                } else if (isMobile && !isModalOpened) {
                    this.setState({
                        isModalOpened: true,
                    });
                }
            } else if (isMobile && isModalOpened) {
                this.setState({
                    isModalOpened: false,
                });
            }
        }

        if (authError) {
            this.redirectToPassport();
        }
    }

    // methods of adding, editing and deleting (validation in components)
    addPassenger = (values: TDataAddPassenger) =>
        this.props.addPassenger(values);

    editPassenger = (values: IPreparedPassenger) =>
        this.props.editPassenger(values);

    removePassenger = (id: string) => this.props.removePassenger({id});

    addDoc = (values: IPassengerDocumentData) => this.props.addDocument(values);

    editDoc = (values: IPassengerDocumentData) =>
        this.props.editDocument(values);

    removeDoc = (id: string) =>
        this.props.removeDocument(id, this.props.userId);

    addCard = (values: IPassengerDocumentData): void =>
        this.props.addCard(values);

    removeCard = (id: string): void =>
        this.props.removeCard(id, this.props.userId);

    refreshPage = (): void => reloadPage();

    goToMain = (): void => this.props.history.push(internalUrl(URLs.index));

    goToPassengersList = (): void => {
        goBack(URLs.accountPassengers);
    };

    // redirect
    redirect(): void {
        const {
            match,
            items,
            history,
            fetched,
            redirectedToJustAdded,
            lastInsertedId,
        } = this.props;

        if (fetched) {
            if (lastInsertedId) {
                if (
                    items.some(
                        (item: {id: string}) => item.id === lastInsertedId,
                    )
                ) {
                    history.push(accountURLs.getPassengerUrl(lastInsertedId));
                }

                redirectedToJustAdded();
            } else if (match.params.userId) {
                if (
                    !items.some(
                        (item: {id: string}) => item.id === match.params.userId,
                    )
                ) {
                    history.push(internalUrl(URLs.accountPassengers));
                }
            }
        }
    }

    redirectToPassport(): void {
        const {
            userInfo: {passportPath},
        } = this.props;
        const href = location.href;

        location.assign(
            getPassportUrl({
                mode: EPassportMode.AUTH,
                passportHost: passportPath,
                retpath: href,
            }),
        );
    }

    renderList(userId: string): ReactNode {
        const {items, docTypes, deviceType, citizenshipForSelect} = this.props;

        return (
            <PassengerList
                items={items}
                userId={userId}
                docTypes={docTypes}
                addPassenger={this.addPassenger}
                deviceType={deviceType}
                citizenship={citizenshipForSelect}
            />
        );
    }

    renderInfo(userId: string): ReactNode {
        const {items, deviceType, fetching} = this.props;
        const {isMobile} = deviceType;

        return (
            <div
                className={cx('info', {
                    info_mobile: isMobile,
                })}
            >
                <PassengerInfo
                    passenger={
                        items.filter(
                            (item: {id: string}) => item.id === userId,
                        )[0]
                    }
                    editPassenger={this.editPassenger}
                    baseLink={URLs.accountPassengers}
                    deviceType={deviceType}
                    fetching={fetching}
                />
                {this.renderDocs(userId)}
                <PassengerRemove
                    onRemoveItem={_partial(this.removePassenger, userId)}
                    isMobile={isMobile}
                />
            </div>
        );
    }

    renderDocs(userId: string): ReactNode {
        const {
            documents,
            docTypes,
            deviceType,
            items,
            citizenshipData,
            citizenshipForSelect,
        } = this.props;

        return (
            <PassengerDocs
                items={documents}
                docTypes={docTypes}
                editDocument={this.editDoc}
                addDocument={this.addDoc}
                removeDocument={this.removeDoc}
                userId={userId}
                passenger={
                    items.filter((item: {id: string}) => item.id === userId)[0]
                }
                deviceType={deviceType}
                citizenship={citizenshipForSelect}
                citizenshipData={citizenshipData}
            />
        );
    }

    renderPage(): ReactNode {
        const {
            items,
            deviceType: {isMobile},
            userId,
        } = this.props;

        if (isMobile || !items.length) {
            return (
                <>
                    {this.renderHelmet(userId)}
                    {this.renderList(userId)}
                </>
            );
        }

        return (
            <>
                {this.renderHelmet(userId)}
                {this.renderList(userId)}
                {this.renderInfo(userId)}
            </>
        );
    }

    renderModal(): ReactNode {
        const {userId} = this.props;
        const {isModalOpened} = this.state;

        return (
            <AccountModal
                isVisible={isModalOpened}
                onClose={this.goToPassengersList}
                backButtonTitle={i18nBlock.infoDotBack()}
                allowFullScreen
                isOwnUrlForModal
                {...prepareQaAttributes('passenger-info')}
            >
                {userId && this.renderInfo(userId)}
            </AccountModal>
        );
    }

    renderError(): ReactNode {
        return (
            <AccountError
                refreshPage={this.refreshPage}
                goToMain={this.goToMain}
            />
        );
    }

    renderHelmet(userId: string): ReactNode {
        const {items} = this.props;
        const passenger = items.filter(
            (item: {id: string}) => item.id === userId,
        )[0];

        return (
            <Helmet>
                <title>
                    {passenger
                        ? i18nCommonBlock.metaDotAccountDotTitleDotUsername({
                              username: passenger.title,
                          })
                        : i18nCommonBlock.metaDotAccountDotTitleDotPassengers()}
                </title>
            </Helmet>
        );
    }

    render(): ReactNode {
        const {
            fetched,
            fetching,
            error,
            deviceType: {isMobile},
        } = this.props;

        return (
            <div
                className={cx('container', {
                    container_mobile: isMobile,
                })}
            >
                {error && this.renderError()}
                {fetched && !fetching && this.renderPage()}
                {fetching && (
                    <Preloader
                        {...prepareQaAttributes({
                            parent: ROOT_QA,
                            current: 'loading',
                        })}
                    />
                )}
                {isMobile && this.renderModal()}
            </div>
        );
    }
}

export default withRouter(
    connect(mapStateToProps, mapDispatchToProps)(AccountPassengerPage),
);
