import React, {PureComponent} from 'react';
import _isEmpty from 'lodash/isEmpty';

import {
    IDocumentEditFormProps,
    IDocumentEditFormState,
} from 'projects/account/pages/Passengers/types';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import getDocumentNameText from 'projects/account/pages/Passengers/components/Documents/utilities/getDocumentNameText';

import PassengerForms from 'projects/account/pages/Passengers/components/PassengerForms/PassengerForms';
import PassengerFormField from 'projects/account/pages/Passengers/components/PassengerFormField/PassengerFormField';
import withFormErrorsDisplay from 'projects/account/components/FormErrorsDisplay/FormErrorsDisplay';

const ROOT_QA = 'document-edit-form';

class DocumentEdit extends PureComponent<
    IDocumentEditFormProps,
    IDocumentEditFormState
> {
    state: IDocumentEditFormState = {
        isVisible: false,
        type: undefined,
        id: '',
    };

    componentDidMount(): void {
        this.updateCommonData();
    }

    componentDidUpdate(): void {
        this.updateCommonData();
    }

    private updateCommonData(): void {
        const {
            type: typeFromProps,
            isVisible: isVisibleFromProps,
            data,
            initialize,
        } = this.props;

        const {
            type: typeFromState,
            isVisible: isVisibleFromState,
            id,
        } = this.state;

        const newState: any = {};
        let newData: any;

        if (isVisibleFromProps && !isVisibleFromState) {
            newState.isVisible = true;

            if (_isEmpty(data) || typeof data === 'undefined') {
                if (typeFromProps !== typeFromState) {
                    newState.type = typeFromProps;
                }

                if (id !== '') {
                    newState.id = '';
                }

                newData = {type: typeFromProps};
            } else if (data.id !== id) {
                newState.id = data.id;
                newState.type = data.type;
                newData = data;
            }
        } else if (isVisibleFromState && !isVisibleFromProps) {
            newState.isVisible = false;
            newState.id = '';
            newData = data;
        }

        if (!_isEmpty(newState)) {
            this.setState(newState, () => initialize(newData));
        }
    }

    private renderField = (item: any): React.ReactNode => {
        const {
            deviceType: {isMobile},
            setFieldRefByName,
            onBlurField,
            onFocusField,
            modalRef,
            canShowErrors,
            citizenship,
        } = this.props;

        const newItem = {...item};

        if (item.type === 'select') {
            if (newItem.name === 'citizenship') {
                newItem.values = citizenship;
            }

            newItem.modalRef = modalRef;
        }

        return (
            <PassengerFormField
                key={newItem.name}
                field={newItem}
                isMobile={isMobile}
                triggerRef={setFieldRefByName(newItem.name)}
                onFocus={onFocusField}
                onBlur={onBlurField}
                canShowErrors={canShowErrors}
                formName="document"
                {...prepareQaAttributes({
                    parent: ROOT_QA,
                    current: newItem.name,
                })}
            />
        );
    };

    render(): React.ReactNode {
        const {
            deviceType: {isMobile},
            onModalClose,
            handleSubmit,
            fields,
        } = this.props;
        const {type} = this.state;

        const title = type ? getDocumentNameText(type) : '';

        return (
            <PassengerForms
                title={title}
                handleSubmit={handleSubmit}
                isMobile={isMobile}
                onModalClose={onModalClose}
                {...prepareQaAttributes(ROOT_QA)}
            >
                {fields.map(this.renderField)}
            </PassengerForms>
        );
    }
}

export default withFormErrorsDisplay<IDocumentEditFormProps>(DocumentEdit);
