import React, {ReactNode, PureComponent} from 'react';
import _isEqual from 'lodash/isEqual';
import {Form as FinalForm} from 'react-final-form';

import {ACCOUNT_PASSENGER_DOCUMENT_FORM} from 'constants/reduxForm';
import {RU_GEO_ID} from 'constants/common';
import {FORM_FIELDS} from 'constants/account';

import {EAccountGoal} from 'utilities/metrika/types/goals/account';
import {ICountry} from 'types/common/ICountry';
import {IPassengerDocumentData} from 'types/common/document/IPassengerDocumentData';
import {
    IBuildFieldProps,
    IErrorsProps,
} from 'projects/account/pages/Passengers/types';

import {IDevice} from 'reducers/common/commonReducerTypes';
import {IPreparedPassenger} from 'reducers/account/passengers/api-types';
import {IPreparedDocument} from 'reducers/account/docs/props';
import {IDocTypesProps} from 'reducers/account/docTypes/api-types';
import {GeoID} from 'reducers/common/countries/reducer';

import {DOCUMENT_OTHER} from 'projects/account/lib/travellers/documents';
import validate from 'projects/account/pages/Passengers/utilities/validateForm/documentEditFormValidate';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {reachGoal} from 'utilities/metrika';
import buildFormFields from 'projects/account/pages/Passengers/utilities/buildFormFields/buildFormFields';

import * as i18nBlock from 'i18n/account';

import AccountModal from 'projects/account/components/AccountModal/AccountModal';
import DocumentEdit from 'projects/account/pages/Passengers/components/Documents/DocumentEdit/DocumentEdit';

import {EDocumentType} from 'server/api/TravelersApi/enums/EDocumentType';

import DocumentsAdd from './DocumentsAdd/DocumentsAdd';
import DocumentsItem from './DocumentsItem/DocumentsItem';

import cx from './Documents.scss';

interface IDocumentsProps {
    deviceType: IDevice;
    userId: string;
    passenger: IPreparedPassenger;
    docTypes: IDocTypesProps;
    items: IPreparedDocument[];
    citizenship: ICountry[];
    citizenshipData: Record<GeoID, ICountry>;
    addDocument: (values: IPassengerDocumentData) => void;
    editDocument: (values: IPassengerDocumentData) => void;
    removeDocument: (id: string) => void;
}

interface IDocumentsState {
    isFormModalOpened: boolean;
    isAddModalOpened: boolean;
    type?: EDocumentType;
    fields: IBuildFieldProps[];
    document?: IPreparedDocument;
    modalRef: React.RefObject<HTMLElement | unknown>;
}

export default class Documents extends PureComponent<
    IDocumentsProps,
    IDocumentsState
> {
    state: IDocumentsState = {
        isFormModalOpened: false,
        isAddModalOpened: false,
        type: undefined,
        fields: [],
        document: undefined,
        modalRef: React.createRef(),
    };

    refModal = React.createRef<HTMLDivElement>();

    private openAddModal = (): void => {
        this.setState({
            isAddModalOpened: true,
        });
    };

    private closeAddModal = (): void =>
        this.setState({
            isAddModalOpened: false,
        });

    private closeFormModal = (): void =>
        this.setState({
            isFormModalOpened: false,
        });

    private openFormModal = (docId?: string, docType?: EDocumentType): void => {
        const {items, docTypes} = this.props;

        let document;
        let type: IDocumentsState['type'];
        let fields: string[] = [];

        if (docId) {
            reachGoal(EAccountGoal.EDIT_DOCUMENT_BUTTON);
            document = items.find(item => item.id === docId);

            if (document) {
                type = document.type;
                fields = Object.keys(FORM_FIELDS).filter(item => {
                    if (FORM_FIELDS[item].documents && type) {
                        return !docTypes.rules[type].unused.includes(item);
                    }

                    return false;
                });
            }
        } else if (docType) {
            reachGoal(EAccountGoal.ADD_DOCUMENT_BUTTON);
            type = docType;
            fields = Object.keys(FORM_FIELDS).filter(item => {
                if (FORM_FIELDS[item].documents && type) {
                    return !docTypes.rules[type].unused.includes(item);
                }

                return false;
            });
        }

        this.setState({
            isFormModalOpened: true,
            document: document,
            type: type,
            fields: buildFormFields(fields),
        });
    };

    private closeBothModals = (): void =>
        this.setState({
            isFormModalOpened: false,
            isAddModalOpened: false,
        });

    getNewValues = (values: IPassengerDocumentData): IPassengerDocumentData => {
        const {type} = this.state;

        const valueCitizenship = Array.isArray(values.citizenship)
            ? values.citizenship[0]
            : values.citizenship;

        return {
            ...values,
            type,
            citizenship: type === DOCUMENT_OTHER ? valueCitizenship : RU_GEO_ID,
        };
    };

    validate = (values: IPassengerDocumentData): IErrorsProps => {
        const {docTypes} = this.props;
        const {type} = this.state;
        const newValues = this.getNewValues(values);
        const {errors} = validate(newValues, docTypes, type);

        return errors;
    };

    handleFormSubmit = (values: IPassengerDocumentData): void => {
        const newValues = this.getNewValues(values);

        this.handleDocumentEdit(newValues);
    };

    private handleDocumentEdit = (formValues: IPassengerDocumentData): void => {
        const {items, addDocument, editDocument, userId} = this.props;

        const values = {
            ...formValues,
            passengerId: userId,
        };

        if (values.id) {
            const docItem = items.find(item => item.id === values.id);

            if (!_isEqual(values, docItem)) {
                editDocument(values);
            }
        } else {
            addDocument(values);
        }

        this.closeBothModals();
    };

    private handleDocumentRemove = (docId: string): void => {
        this.props.removeDocument(docId);
    };

    private renderModal(): ReactNode {
        const {deviceType, citizenship, docTypes} = this.props;
        const {type, fields, document, isFormModalOpened} = this.state;

        return (
            <AccountModal
                isVisible={isFormModalOpened}
                onClose={this.closeFormModal}
                allowFullScreen
            >
                <div className={cx('modalContent')} ref={this.refModal}>
                    <FinalForm
                        onSubmit={this.handleFormSubmit}
                        validate={this.validate}
                    >
                        {({
                            handleSubmit,
                            submitFailed,
                            errors,
                            values,
                            form,
                        }): ReactNode => (
                            <DocumentEdit
                                initialize={form.initialize}
                                handleSubmit={handleSubmit}
                                canShowErrors={submitFailed}
                                formValues={values}
                                formErrors={errors}
                                modalRef={this.refModal.current}
                                form={ACCOUNT_PASSENGER_DOCUMENT_FORM}
                                deviceType={deviceType}
                                isVisible={isFormModalOpened}
                                onModalClose={this.closeFormModal}
                                citizenship={citizenship}
                                docTypes={docTypes}
                                data={document}
                                fields={fields}
                                type={type}
                                finalFormManager
                            />
                        )}
                    </FinalForm>
                </div>
            </AccountModal>
        );
    }

    private renderDocuments(): ReactNode {
        const {
            items,
            docTypes,
            deviceType: {isMobile},
            citizenshipData,
        } = this.props;

        if (items.length) {
            return (
                <div className={cx('items')}>
                    {items.map((item, index) => {
                        let citizenshipDataValue: ICountry | undefined;

                        if (item.citizenship) {
                            citizenshipDataValue =
                                citizenshipData[item.citizenship];
                        }

                        return (
                            <DocumentsItem
                                key={item.id}
                                openModal={this.openFormModal}
                                removeItem={this.handleDocumentRemove}
                                document={item}
                                docTypes={docTypes.items}
                                isMobile={isMobile}
                                citizenshipData={citizenshipDataValue}
                                {...prepareQaAttributes({
                                    key: index,
                                    current: 'document-item',
                                })}
                            />
                        );
                    })}
                </div>
            );
        }

        return (
            <div
                className={cx('empty')}
                {...prepareQaAttributes('documents-empty-text')}
            >
                {i18nBlock.docsDotEmpty()}
            </div>
        );
    }

    private renderAdd(): ReactNode {
        const {
            docTypes,
            deviceType: {isMobile},
            passenger,
        } = this.props;
        const {isAddModalOpened} = this.state;

        return (
            <DocumentsAdd
                isModalOpened={isAddModalOpened}
                onOpenFormModal={this.openFormModal}
                onOpenModal={this.openAddModal}
                onCloseModal={this.closeAddModal}
                docTypes={docTypes.items}
                isMobile={isMobile}
                passenger={passenger}
            />
        );
    }

    render(): ReactNode {
        return (
            <div className={cx('docs')}>
                {this.renderDocuments()}
                {this.renderAdd()}
                {this.renderModal()}
            </div>
        );
    }
}
