import React, {PureComponent} from 'react';
import _get from 'lodash/get';

import {IPreparedPassenger} from 'reducers/account/passengers/api-types';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import getDocumentNameText from 'projects/account/pages/Passengers/components/Documents/utilities/getDocumentNameText';

import * as i18nAccountBlock from 'i18n/account';

import Button from 'components/Button/Button';
import {Intent} from 'components/Intent/Intent';
import AccountModal from 'projects/account/components/AccountModal/AccountModal';
import PlusIcon from 'icons/16/Plus';
import KebabVerticalIcon from 'icons/16/KebabVertical';

import {EDocumentType} from 'server/api/TravelersApi/enums/EDocumentType';

import cx from './DocumentsAdd.scss';

interface IDocumentsAddProps {
    isModalOpened: boolean;
    docTypes: EDocumentType[];
    isMobile: boolean;
    passenger?: IPreparedPassenger;
    onOpenFormModal: (id?: string, docType?: EDocumentType) => void;
    onCloseModal: () => void;
    onOpenModal: () => void;
}

interface IDocumentsAddState {
    isShowAll: boolean;
}

export default class DocumentsAdd extends PureComponent<
    IDocumentsAddProps,
    IDocumentsAddState
> {
    state = {
        isShowAll: false,
    };

    private handleShowAll = (): void =>
        this.setState({
            isShowAll: true,
        });

    private handleFormModalOpen = (docType: EDocumentType) => (): void => {
        const {onOpenFormModal} = this.props;

        onOpenFormModal(undefined, docType);
    };

    private renderMoreButton(): React.ReactNode {
        const {isMobile} = this.props;
        const {isShowAll} = this.state;

        if (!isShowAll && !isMobile) {
            return (
                <div className={cx('item')}>
                    <Intent
                        size={isMobile ? 'l' : 'm-inset'}
                        onClick={this.handleShowAll}
                        icon={<KebabVerticalIcon />}
                        {...prepareQaAttributes('document-add-more')}
                    />
                </div>
            );
        }

        return null;
    }

    render(): React.ReactNode {
        const {
            docTypes,
            isMobile,
            passenger,
            isModalOpened,
            onOpenModal,
            onCloseModal,
        } = this.props;
        const {isShowAll} = this.state;

        const passengerTitle = _get(passenger, 'title', '');

        let docTypesReady = docTypes.map((item, index) => (
            <div key={item} className={cx('item')}>
                <Intent
                    size={isMobile ? 'l' : 'm-inset'}
                    text={getDocumentNameText(item)}
                    onClick={this.handleFormModalOpen(item)}
                    {...prepareQaAttributes({
                        current: `document-add-button`,
                        key: index,
                    })}
                />
            </div>
        ));

        if (!isShowAll && !isMobile) {
            docTypesReady = docTypesReady.slice(0, 3);
        }

        if (isMobile) {
            return (
                <div className={cx('add')}>
                    <Button
                        onClick={onOpenModal}
                        width="max"
                        size="l"
                        className={cx('mobileButton')}
                        icon={<PlusIcon />}
                        {...prepareQaAttributes('document-add-more')}
                    >
                        {i18nAccountBlock.docsDotAdd()}
                    </Button>

                    <AccountModal
                        isVisible={isModalOpened}
                        onClose={onCloseModal}
                        title={passengerTitle}
                        allowFullScreen
                    >
                        <div className={cx('block_mobile')}>
                            {docTypesReady}
                        </div>
                    </AccountModal>
                </div>
            );
        }

        return (
            <div className={cx('add')}>
                <div className={cx('title')}>
                    {i18nAccountBlock.docsDotAddDotTitle()}
                </div>
                <div className={cx('block')}>
                    {docTypesReady}
                    {this.renderMoreButton()}
                </div>
            </div>
        );
    }
}
