import {ReactNode, PureComponent} from 'react';

import {ICountry} from 'types/common/ICountry';

import {IPreparedDocument} from 'reducers/account/docs/props';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import getDocumentNameText from 'projects/account/pages/Passengers/components/Documents/utilities/getDocumentNameText';

import * as i18nAccountBlock from 'i18n/account';

import FlagIcon from 'components/FlagIcon/FlagIcon';
import Meta from 'projects/account/pages/Passengers/components/Meta/Meta';
import Actions from 'projects/account/pages/Passengers/components/PassengerActions/PassengerActions';
import Preloader from 'projects/account/components/AccountPreloader/AccountPreloader';
import RemoveDialog from 'projects/account/pages/Passengers/components/RemoveDialog/RemoveDialog';
import WarningFilledIcon from 'icons/12/WarningFilled';

import {EDocumentType} from 'server/api/TravelersApi/enums/EDocumentType';

import cx from './DocumentsItem.scss';

interface IDocumentsItemProps extends IWithQaAttributes {
    document: IPreparedDocument;
    citizenshipData?: ICountry;
    docTypes: EDocumentType[];
    isMobile: boolean;
    openModal: (id: string) => void;
    removeItem: (id: string) => void;
}

interface IDocumentsItemState {
    isDeleting: boolean;
}

export default class DocumentsItem extends PureComponent<
    IDocumentsItemProps,
    IDocumentsItemState
> {
    state = {
        isDeleting: false,
    };

    private handleModalOpen = (): void => {
        const {
            openModal,
            document: {id},
        } = this.props;

        openModal(id);
    };

    private handleDeleting = (): void =>
        this.setState({
            isDeleting: true,
        });

    private handleClose = (): void =>
        this.setState({
            isDeleting: false,
        });

    private handleDelete = (): void => {
        const {
            removeItem,
            document: {id},
        } = this.props;

        this.setState(
            {
                isDeleting: false,
            },
            () => removeItem(id),
        );
    };

    private renderDocumentName(): ReactNode {
        const {
            document: {type, isExpired},
        } = this.props;

        const documentName = getDocumentNameText(type);

        return (
            <div className={cx('docProps')}>
                <div
                    className={cx('type')}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'type',
                    })}
                >
                    {documentName}
                </div>
                {isExpired && this.renderExpired()}
            </div>
        );
    }

    private renderExpired(): ReactNode {
        return (
            <div className={cx('expired')}>
                <WarningFilledIcon className={cx('expiredIcon')} />
                {i18nAccountBlock.docsDotExpired()}
            </div>
        );
    }

    private renderIcons(): ReactNode {
        return (
            <div className={cx('icons')}>
                <div className={cx('icon')}>
                    <Actions
                        icon="edit"
                        onClick={this.handleModalOpen}
                        {...prepareQaAttributes({
                            parent: this.props,
                            current: 'edit',
                        })}
                    />
                </div>
                <div className={cx('icon')}>
                    <Actions
                        icon="delete"
                        onClick={this.handleDeleting}
                        {...prepareQaAttributes({
                            parent: this.props,
                            current: 'remove',
                        })}
                    />
                </div>
            </div>
        );
    }

    private renderName(): ReactNode {
        const {
            document: {
                firstName = '',
                middleName = '',
                lastName = '',
                firstNameEn = '',
                middleNameEn = '',
                lastNameEn = '',
            },
        } = this.props;

        const fullNameRu = i18nAccountBlock.docsDotFullName({
            lastName,
            firstName,
            middleName,
        });
        const fullNameEn = i18nAccountBlock.docsDotFullName({
            lastName: lastNameEn,
            firstName: firstNameEn,
            middleName: middleNameEn,
        });

        return (
            <div className={cx('passengerName')}>
                <div
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'name-big',
                    })}
                >
                    {fullNameRu || fullNameEn}
                </div>

                {Boolean(fullNameRu && fullNameEn) && (
                    <div
                        className={cx('small')}
                        {...prepareQaAttributes({
                            parent: this.props,
                            current: 'name-small',
                        })}
                    >
                        {fullNameEn}
                    </div>
                )}
            </div>
        );
    }

    private renderMeta(): ReactNode {
        const {
            isMobile,
            document: {citizenship, number, expirationDate},

            citizenshipData,
        } = this.props;

        return (
            <div className={cx('meta')}>
                <Meta
                    name={i18nAccountBlock.docsDotNumber()}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'number',
                    })}
                >
                    {number}
                </Meta>
                {citizenship && citizenshipData && (
                    <Meta
                        name={i18nAccountBlock.docsDotCitizenship()}
                        className={cx({
                            metaItem: isMobile,
                        })}
                        {...prepareQaAttributes({
                            parent: this.props,
                            current: 'citizenship',
                        })}
                    >
                        <div className={cx('citizenship')}>
                            {citizenshipData.code2 && (
                                <div className={cx('flag')}>
                                    <FlagIcon
                                        country={citizenshipData.code2.toLowerCase()}
                                    />
                                </div>
                            )}
                            {citizenshipData.title}
                        </div>
                    </Meta>
                )}
                {expirationDate && (
                    <Meta
                        name={i18nAccountBlock.docsDotExpirationDate()}
                        className={cx('expirationDate')}
                        {...prepareQaAttributes({
                            parent: this.props,
                            current: 'expiration-date',
                        })}
                    >
                        {expirationDate}
                    </Meta>
                )}
            </div>
        );
    }

    private renderLoader(): ReactNode {
        const {
            document: {isLoading},
        } = this.props;

        if (isLoading) {
            return (
                <Preloader
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'loading',
                    })}
                    fixed={false}
                />
            );
        }

        return null;
    }

    private renderRemove(): ReactNode {
        const {isDeleting} = this.state;
        const {
            document: {type},
            isMobile,
        } = this.props;

        const documentName = getDocumentNameText(type);
        const documentTitleLower =
            documentName.substr(0, 1).toLowerCase() + documentName.substr(1);

        if (isDeleting) {
            return (
                <RemoveDialog
                    isAbsolute={true}
                    title={i18nAccountBlock.docsDotDeleteDotDialog({
                        document: documentTitleLower,
                    })}
                    onClose={this.handleClose}
                    onSubmit={this.handleDelete}
                    isMobile={isMobile}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'dialog',
                    })}
                />
            );
        }

        return null;
    }

    render(): ReactNode {
        const {
            isMobile,
            document: {isExpired},
        } = this.props;

        return (
            <div
                className={cx('item', {
                    item_expired: isExpired,
                    item_mobile: isMobile,
                })}
                {...prepareQaAttributes(this.props)}
            >
                <div className={cx('block')}>
                    {this.renderDocumentName()}
                    {this.renderIcons()}
                </div>
                {this.renderName()}
                {this.renderMeta()}
                {this.renderLoader()}
                {this.renderRemove()}
            </div>
        );
    }
}
