import {PureComponent, ReactNode} from 'react';

import {RU_GEO_ID} from 'constants/common';

import {
    IAddPassengerFormProps,
    IPassengerFormsDefaultState,
} from 'projects/account/pages/Passengers/types';

import {DOCUMENT_OTHER} from 'projects/account/lib/travellers/documents';
import buildFormFields from 'projects/account/pages/Passengers/utilities/buildFormFields/buildFormFields';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import * as i18nAccountFormsBlock from 'i18n/accountForms';
import * as i18nBlock from 'i18n/account';

import PassengerForms from 'projects/account/pages/Passengers/components/PassengerForms/PassengerForms';
import PassengerFormField from 'projects/account/pages/Passengers/components/PassengerFormField/PassengerFormField';
import FormErrorsDisplay from 'projects/account/components/FormErrorsDisplay/FormErrorsDisplay';

const DEFAULT_FIELDS = [
    'lastName',
    'firstName',
    'middleName',
    'birthDate',
    'gender',
    'citizenship',
    'type',
    'number',
    'itn',
    'email',
    'phone',
    'train_notifications_enabled',
];
const OPTIONS_FOR_FIELDS = {
    itn: {
        size: 'big',
    },
};

const ROOT_QA = 'passenger-add-form';

interface IPassengerAddFormState extends IPassengerFormsDefaultState {
    docTypes: string[];
}

class PassengerAdd extends PureComponent<
    IAddPassengerFormProps,
    IPassengerAddFormState
> {
    state: IPassengerAddFormState = {
        fields: buildFormFields(DEFAULT_FIELDS, OPTIONS_FOR_FIELDS),
        docTypes: [],
    };

    static getDerivedStateFromProps(
        nextProps: IAddPassengerFormProps,
        prevState: IPassengerAddFormState,
    ): IPassengerAddFormState {
        if (prevState.docTypes.length === 0) {
            return {
                ...prevState,
                docTypes: nextProps.docTypes.items,
            };
        }

        return prevState;
    }

    private handleDocTypeChange = (newDocType: string[]): void => {
        const {docTypes} = this.props;
        const newFields = [...DEFAULT_FIELDS];

        if (!docTypes.rules[newDocType[0]].unused.includes('expirationDate')) {
            const documentNumberIndex = newFields.findIndex(
                fieldName => fieldName === 'number',
            );

            newFields.splice(documentNumberIndex + 1, 0, 'expirationDate');
        }

        this.setState({
            fields: buildFormFields(newFields, OPTIONS_FOR_FIELDS),
        });
    };

    private handleCitizenshipChange = (newCitizenship: number[]): void => {
        const {docTypes, change} = this.props;
        const citizenship = newCitizenship[0];

        const newDocTypes = docTypes.items.filter((item: string) => {
            if (citizenship === RU_GEO_ID) {
                return item !== DOCUMENT_OTHER;
            }

            return item === DOCUMENT_OTHER;
        });

        this.setState(
            {
                docTypes: newDocTypes,
            },
            () => {
                if (newDocTypes.length === 1) {
                    change('type', newDocTypes[0]);
                    this.handleDocTypeChange([newDocTypes[0]]);
                } else {
                    change('type', undefined);
                }
            },
        );
    };

    private renderField = (item: any): ReactNode => {
        const {
            deviceType: {isMobile},
            modalRef,
            setFieldRefByName,
            onFocusField,
            onBlurField,
            canShowErrors,
            citizenship,
        } = this.props;

        const newItem = {...item};

        if (item.type === 'select') {
            if (newItem.name === 'citizenship') {
                newItem.values = citizenship;
                newItem.onChange = this.handleCitizenshipChange;
            } else if (newItem.name === 'type') {
                newItem.values = this.state.docTypes;
                newItem.onChange = this.handleDocTypeChange;
            }

            newItem.modalRef = modalRef;
        }

        if (item.type === 'radio') {
            newItem.options = [
                {
                    value: 'male',
                    title: i18nAccountFormsBlock.radioDotGenderDotMale(),
                },
                {
                    value: 'female',
                    title: i18nAccountFormsBlock.radioDotGenderDotFemale(),
                },
            ];
        }

        if (item.type === 'checkbox') {
            newItem.options = {
                title: i18nAccountFormsBlock.labelsDotTrainNotificationsEnabled(),
            };
        }

        return (
            <PassengerFormField
                key={newItem.name}
                field={newItem}
                isMobile={isMobile}
                triggerRef={setFieldRefByName(newItem.name)}
                onFocus={onFocusField}
                onBlur={onBlurField}
                canShowErrors={canShowErrors}
                formName="passenger"
                {...prepareQaAttributes({
                    parent: ROOT_QA,
                    current: newItem.name,
                })}
            />
        );
    };

    render(): ReactNode {
        const {
            deviceType: {isMobile},
            onModalClose,
            handleSubmit,
        } = this.props;
        const {fields} = this.state;

        return (
            <PassengerForms
                title={i18nBlock.infoDotEditDotTitle()}
                handleSubmit={handleSubmit}
                onModalClose={onModalClose}
                isMobile={isMobile}
                {...prepareQaAttributes(ROOT_QA)}
            >
                {fields.map(this.renderField)}
            </PassengerForms>
        );
    }
}

// eslint-disable-next-line new-cap
export default FormErrorsDisplay<IAddPassengerFormProps>(PassengerAdd);
