import {PureComponent, ReactNode} from 'react';

import {IPassengerEditFormProps} from 'projects/account/pages/Passengers/types';

import buildFormFields from 'projects/account/pages/Passengers/utilities/buildFormFields/buildFormFields';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import * as i18nAccountBlock from 'i18n/account';
import * as i18nBlock from 'i18n/accountForms';

import FormErrorsDisplay from 'projects/account/components/FormErrorsDisplay/FormErrorsDisplay';
import PassengerFormField from 'projects/account/pages/Passengers/components/PassengerFormField/PassengerFormField';
import PassengerForms from 'projects/account/pages/Passengers/components/PassengerForms/PassengerForms';

const OPTIONS_FOR_FIELDS = {
    itn: {
        size: 'middle',
    },
};
const DEFAULT_FIELDS = buildFormFields(
    [
        'title',
        'birthDate',
        'gender',
        'itn',
        'email',
        'phone',
        'train_notifications_enabled',
    ],
    OPTIONS_FOR_FIELDS,
);

const ROOT_QA = 'passenger-edit-form';

interface IPassengerEditState {
    id: string;
    isVisible: boolean;
}

class PassengerEdit extends PureComponent<
    IPassengerEditFormProps,
    IPassengerEditState
> {
    state = {
        isVisible: false,
        id: '',
    };

    componentDidMount(): void {
        this.updateData();
    }

    componentDidUpdate(): void {
        this.updateData();
    }

    private updateData(): void {
        const {isVisible: isVisibleFromProps, data, initialize} = this.props;
        const {isVisible: isVisibleFromState, id} = this.state;

        if (isVisibleFromProps && !isVisibleFromState) {
            if (data && data.id !== id) {
                this.setState(
                    {
                        isVisible: true,
                        id: data.id,
                    },
                    () => initialize(data),
                );
            }
        } else if (isVisibleFromState && !isVisibleFromProps) {
            this.setState({
                id: '',
                isVisible: false,
            });
        }
    }

    private renderField = (item: any): ReactNode => {
        const {
            deviceType: {isMobile},
            setFieldRefByName,
            onBlurField,
            onFocusField,
            canShowErrors,
        } = this.props;

        const newItem = {...item};

        if (item.type === 'radio') {
            newItem.options = [
                {
                    value: 'male',
                    title: i18nBlock.radioDotGenderDotMale(),
                },
                {
                    value: 'female',
                    title: i18nBlock.radioDotGenderDotFemale(),
                },
            ];
        }

        if (item.type === 'checkbox') {
            newItem.options = {
                title: i18nBlock.labelsDotTrainNotificationsEnabled(),
            };
        }

        return (
            <PassengerFormField
                key={newItem.name}
                field={newItem}
                isMobile={isMobile}
                triggerRef={setFieldRefByName(newItem.name)}
                onFocus={onFocusField}
                onBlur={onBlurField}
                canShowErrors={canShowErrors}
                formName="passenger-edit"
                {...prepareQaAttributes({
                    parent: ROOT_QA,
                    current: newItem.name,
                })}
            />
        );
    };

    render(): ReactNode {
        const {
            deviceType: {isMobile},
            onModalClose,
            handleSubmit,
        } = this.props;

        return (
            <PassengerForms
                title={i18nAccountBlock.infoDotEditDotTitle()}
                handleSubmit={handleSubmit}
                onModalClose={onModalClose}
                isMobile={isMobile}
                {...prepareQaAttributes(ROOT_QA)}
            >
                {DEFAULT_FIELDS.map(this.renderField)}
            </PassengerForms>
        );
    }
}

// eslint-disable-next-line new-cap
export default FormErrorsDisplay<IPassengerEditFormProps>(PassengerEdit);
