import React, {ReactNode, PureComponent} from 'react';
import {Field, FieldRenderProps} from 'react-final-form';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import getFieldLabelText from 'projects/account/pages/Passengers/components/PassengerFormField/utilities/getFieldLabelText';

import * as i18nBlock from 'i18n/accountForms';

import RadioButton, {
    ERadioButtonState,
} from 'components/RadioButton/RadioButton';
import Checkbox from 'components/Checkbox/Checkbox';

import PassengerInput from './blocks/PassengerFormInput/PassengerFormInput';
import PassengerFormSelect from './blocks/PassengerFormSelect/PassengerFormSelect';

import cx from './PassengerFormField.scss';

// Хак для final-form, чтобы он не удалял из стэйта поля с пустыми значениями
const parse = (value: unknown): unknown => value;

interface IFieldProps {
    type: string;
    size: string;
    sizeInput?: string;
    name: string;
    small?: boolean;
    values?: [string[]] | string[];
    options?: any;
    onChange?: () => void;
    modalRef: any;
}

interface IPassengerFormFieldProps extends IWithQaAttributes {
    field: IFieldProps;
    isMobile: boolean;
    triggerRef: any;
    canShowErrors: boolean;
    onFocus?: React.FocusEventHandler<HTMLElement>;
    onBlur?: React.FocusEventHandler<HTMLElement>;
    formName: string;
}

interface IExtWrappedFieldProps extends FieldRenderProps<string> {
    modalRef: any;
}

export default class PassengerFormField extends PureComponent<IPassengerFormFieldProps> {
    private onInputChange =
        (handleChange: FieldRenderProps<string>['input']['onChange']) =>
        (_: any, value: string): void =>
            handleChange(value);

    private onCheckBoxChange =
        (handleChange: FieldRenderProps<boolean>['input']['onChange']) =>
        (e: React.ChangeEvent<HTMLInputElement>): void =>
            handleChange(e.target.checked);

    private getInputFieldState(error: boolean): 'error' | undefined {
        const {canShowErrors} = this.props;

        return error && canShowErrors ? 'error' : undefined;
    }

    private getRadioFieldState(error: boolean): ERadioButtonState {
        const {canShowErrors} = this.props;

        return error && canShowErrors ? RadioButton.ERROR : RadioButton.NORMAL;
    }

    private renderField(): ReactNode {
        const {field} = this.props;

        switch (field.type) {
            case 'text':
                return this.renderText();
            case 'date':
                return this.renderDate();
            case 'select':
                return this.renderSelect();
            case 'radio':
                return this.renderRadio();
            case 'checkbox':
                return this.renderCheckbox();
            default:
                return null;
        }
    }

    private renderText(): ReactNode {
        const {field} = this.props;

        return (
            <Field
                name={field.name}
                component={this.renderInputComponent}
                type={field.type}
                placeholder=""
                parse={parse}
            />
        );
    }

    private renderDate(): ReactNode {
        const {field} = this.props;

        return (
            <Field
                name={field.name}
                component={this.renderInputComponent}
                placeholder={i18nBlock.placeholderDotDate()}
                type={field.type}
                parse={parse}
            />
        );
    }

    private renderSelect(): ReactNode {
        const {field} = this.props;

        return (
            <Field
                name={field.name}
                component={this.renderSelectComponent}
                placeholder={i18nBlock.placeholderDotSelect()}
                type={field.type}
                values={field.values}
                onChange={field.onChange}
                modalRef={field.modalRef}
            />
        );
    }

    private renderRadio(): ReactNode {
        const {field} = this.props;

        return (
            <Field
                name={field.name}
                component={this.renderRadioComponent}
                options={field.options}
            />
        );
    }

    private renderCheckbox(): ReactNode {
        const {field} = this.props;

        return (
            <Field
                name={field.name}
                component={this.renderCheckboxComponent}
                options={field.options}
            />
        );
    }

    private renderInputComponent = (props: FieldRenderProps<string>) => {
        const {field, triggerRef, isMobile} = this.props;

        return (
            <div>
                <PassengerInput
                    {...props.input}
                    id={`${this.props.formName}-${props.input.name}`}
                    type={field.type}
                    size={isMobile ? 'l' : 'm'}
                    onChange={this.onInputChange(props.input.onChange)}
                    inputRef={triggerRef}
                    onFocus={this.props.onFocus}
                    onBlur={this.props.onBlur}
                    state={this.getInputFieldState(props.meta.error)}
                    placeholder={props.placeholder}
                    {...prepareQaAttributes(this.props)}
                />
            </div>
        );
    };

    private renderRadioComponent = (
        props: FieldRenderProps<string> & {
            options: {value: string; name: string; title: string}[];
        },
    ) => (
        <RadioButton
            {...props.input}
            size={this.props.isMobile ? 'l' : 'm'}
            state={this.getRadioFieldState(props.meta.error)}
            id={`${this.props.formName}-${props.input.name}`}
            width="max"
            radioRef={this.props.triggerRef}
            onFocus={this.props.onFocus}
            onBlur={this.props.onBlur}
            {...prepareQaAttributes(this.props)}
        >
            {props.options.map(item => {
                return (
                    <RadioButton.Radio
                        key={item.value}
                        value={item.value}
                        controlAttrs={prepareQaAttributes({
                            parent: this.props,
                            current: item.value,
                        })}
                    >
                        {item.title}
                    </RadioButton.Radio>
                );
            })}
        </RadioButton>
    );

    private renderSelectComponent = (props: IExtWrappedFieldProps) => (
        <PassengerFormSelect
            {...props}
            id={`${this.props.formName}-${props.input.name}`}
            selectRef={this.props.triggerRef}
            onFocus={this.props.onFocus}
            onBlur={this.props.onBlur}
            modalRef={props.modalRef}
            hasError={props.meta.error && this.props.canShowErrors}
            isMobile={this.props.isMobile}
            {...prepareQaAttributes(this.props)}
        />
    );

    private renderCheckboxComponent = (
        props: FieldRenderProps<boolean> & {
            options: {title: string};
        },
    ) => (
        <Checkbox
            label={props.options.title}
            size="m"
            checked={props.input.value}
            onChange={this.onCheckBoxChange(props.input.onChange)}
            {...prepareQaAttributes(this.props)}
        />
    );

    render(): ReactNode {
        const {formName, field, isMobile} = this.props;

        if (field.type === 'checkbox') {
            return (
                <div
                    className={cx('field', `field_${field.size}`, {
                        field_mobile: isMobile,
                    })}
                    data-form-field
                >
                    {this.renderField()}
                </div>
            );
        }

        return (
            <div
                className={cx('field', `field_${field.size}`, {
                    field_mobile: isMobile,
                })}
                data-form-field
            >
                <div
                    className={cx(
                        'field_input',
                        field.sizeInput && `field_${field.sizeInput}`,
                    )}
                >
                    <label
                        className={cx('label')}
                        htmlFor={`${formName}-${field.name}`}
                    >
                        {getFieldLabelText(field.name)}
                    </label>
                    {this.renderField()}
                </div>
            </div>
        );
    }
}
