import React, {createRef} from 'react';
import _noop from 'lodash/noop';
import {FieldInputProps} from 'react-final-form';

import {ICountry} from 'types/common/ICountry';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import getDocumentNameText from 'projects/account/pages/Passengers/components/Documents/utilities/getDocumentNameText';

import Select, {ISelectOption} from 'components/Select/Select';
import FlagIcon from 'components/FlagIcon/FlagIcon';

import ScopeContext from 'contexts/ScopeContext';

import {isDocumentType} from 'server/api/TravelersApi/enums/EDocumentType';

import cx from './PassengerFormSelect.scss';

import './Select_custom.css';

const MENU_MAX_HEIGHT = 240;

interface IPassengerFormSelectProps<Value extends string>
    extends IWithQaAttributes {
    id: string;
    input: FieldInputProps<Value>;
    placeholder?: string;
    hasError: boolean;
    isMobile: boolean;
    fieldRef?: React.RefObject<HTMLElement>;
    modalRef: React.RefObject<HTMLElement>;
    values: Value[];
    selectRef: React.RefObject<HTMLDivElement>;
    onBlur: React.FocusEventHandler;
    onFocus: React.FocusEventHandler;
    onChange: (values: Value[]) => void;
    tabIndex: number;
}

export default class PassengerFormSelect<
    Value extends string,
> extends React.Component<IPassengerFormSelectProps<Value>> {
    fieldRef = createRef<HTMLDivElement>();

    static defaultProps = {
        values: [],
        selectRef: _noop,
        onBlur: _noop,
        onFocus: _noop,
        onChange: _noop,
        tabIndex: -1,
    };

    componentDidMount(): void {
        this.forceUpdate();
    }

    renderTypeItem({
        data,
    }: ISelectOption<Value, string | number>): React.ReactNode {
        const itemText = isDocumentType(data) ? getDocumentNameText(data) : '';

        return (
            <div className={cx('type-item')} title={itemText}>
                {itemText}
            </div>
        );
    }

    renderCitizenshipItem({
        data,
    }: ISelectOption<Value, ICountry>): React.ReactNode {
        const itemText = data.title;

        return (
            <div className={cx('citizenship-item')} title={itemText}>
                {data.code2 && (
                    <span className={cx('icon')}>
                        <FlagIcon
                            width={16}
                            height={12}
                            country={data.code2.toLowerCase()}
                        />
                    </span>
                )}
                {itemText}
            </div>
        );
    }

    toSelectItem = (
        value: string | number | typeof Select.Separator | ICountry,
    ):
        | ISelectOption<number | string, ICountry | string | number>
        | typeof Select.Separator => {
        if (value === Select.Separator) {
            return value;
        }

        if (this.props.input.name === 'citizenship') {
            const citizenshipValue = value as ICountry;

            return {
                value: citizenshipValue.geoId,
                data: citizenshipValue,
            };
        }

        const stringValue = value as string;

        if (this.props.input.name === 'type') {
            return {
                value: stringValue,
                data: stringValue,
            };
        }

        return {value: stringValue, data: value};
    };

    renderItem = (
        option: ISelectOption<Value, ICountry | string | number>,
    ): React.ReactNode => {
        if (this.props.input.name === 'citizenship') {
            return this.renderCitizenshipItem(
                option as ISelectOption<Value, ICountry>,
            );
        }

        return this.renderTypeItem(
            option as ISelectOption<Value, string | number>,
        );
    };

    handleChange = (value: Value): void => {
        this.props.input.onChange([value]);
        this.props.onChange([value]);
    };

    render(): React.ReactNode {
        const {
            input,
            placeholder,
            hasError,
            isMobile,
            values,
            selectRef,
            onFocus,
            onBlur,
            tabIndex,
        } = this.props;

        const options = values.map(this.toSelectItem);
        const value = ([] as string[]).concat(input.value)[0];

        return (
            <div
                className={cx('wrap')}
                ref={selectRef}
                onFocus={onFocus}
                onBlur={onBlur}
                tabIndex={tabIndex}
            >
                <div ref={this.fieldRef} style={{position: 'relative'}} />
                {this.fieldRef.current && (
                    <ScopeContext.Consumer>
                        {(scope): React.ReactNode => (
                            <Select
                                {...input}
                                id={this.props.id}
                                error={hasError}
                                menuMaxHeight={MENU_MAX_HEIGHT}
                                menuWidth="fixed"
                                onChange={this.handleChange}
                                options={options}
                                placeholder={placeholder}
                                renderItem={this.renderItem}
                                renderValue={this.renderItem}
                                size={isMobile ? 'l' : 'm-inset'}
                                theme="outlined"
                                value={value}
                                width="max"
                                scope={scope ? {current: scope} : undefined}
                                virtualized={!isMobile}
                                {...prepareQaAttributes(this.props)}
                            />
                        )}
                    </ScopeContext.Consumer>
                )}
            </div>
        );
    }
}
