import React, {PureComponent} from 'react';
import {withRouter, RouteComponentProps} from 'react-router-dom';
import _isEqual from 'lodash/isEqual';
import {Form as FinalForm} from 'react-final-form';

import {ACCOUNT_PASSENGER_INDEX_FORM} from 'constants/reduxForm';

import {IDevice} from 'reducers/common/commonReducerTypes';
import {IPreparedPassenger} from 'reducers/account/passengers/api-types';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import validate from 'projects/account/pages/Passengers/utilities/validateForm/passengerEditFormValidate';
import getGenderText from 'projects/account/pages/Passengers/components/PassengerInfo/utilities/getGenderText';

import * as i18nBlock from 'i18n/account-PassengerInfo';

import Actions from 'projects/account/pages/Passengers/components/PassengerActions/PassengerActions';
import Meta from 'projects/account/pages/Passengers/components/Meta/Meta';
import PassengerEdit from 'projects/account/pages/Passengers/components/PassengerEdit/PassengerEdit';
import Preloader from 'projects/account/components/AccountPreloader/AccountPreloader';
import AccountModal from 'projects/account/components/AccountModal/AccountModal';

import cx from './PassengerInfo.scss';

interface IPassengerInfoProps extends RouteComponentProps {
    passenger: IPreparedPassenger;
    editPassenger: (values: IPreparedPassenger) => void;
    baseLink: string;
    deviceType: IDevice;
    fetching: boolean;
}

interface IPassengerInfoState {
    isModalOpened: boolean;
}

class PassengerInfo extends PureComponent<
    IPassengerInfoProps,
    IPassengerInfoState
> {
    state = {
        isModalOpened: false,
    };

    refModal = React.createRef<HTMLDivElement>();

    private openModal = () => {
        this.setState({
            isModalOpened: true,
        });
    };

    private closeModal = () => {
        this.setState({
            isModalOpened: false,
        });
    };

    private editItem = (values: IPreparedPassenger) => {
        if (!_isEqual(values, this.props.passenger)) {
            this.props.editPassenger(values);
        }

        this.closeModal();
    };

    private renderNick() {
        const {passenger} = this.props;

        return (
            <div className={cx('block')}>
                <div className={cx('nickBlock')}>
                    <div
                        className={cx('passengerName')}
                        {...prepareQaAttributes('passenger-info-title')}
                    >
                        {passenger.title}
                    </div>
                    <Actions
                        icon="edit"
                        onClick={this.openModal}
                        {...prepareQaAttributes('passenger-edit-button')}
                    />
                </div>
                <div className={cx('small')}>
                    {i18nBlock.nickDotDisclaimer()}
                </div>
            </div>
        );
    }

    validate = (values: IPreparedPassenger) => {
        const {errors} = validate(values);

        return errors;
    };

    private renderMeta() {
        const {
            deviceType: {isMobile},
            passenger,
        } = this.props;

        return (
            <div className={cx('meta', isMobile && 'meta_mobile')}>
                {(passenger.birthDate || passenger.gender) && (
                    <div className={cx('metaBlock')}>
                        {passenger.birthDate && (
                            <Meta
                                name={i18nBlock.birthdate()}
                                className={cx('metaItem')}
                                {...prepareQaAttributes(
                                    'passenger-info-birthdate',
                                )}
                            >
                                {passenger.birthDate}
                            </Meta>
                        )}
                        {passenger.gender && (
                            <Meta
                                name={i18nBlock.gender()}
                                className={cx('metaItem')}
                                {...prepareQaAttributes(
                                    'passenger-info-gender',
                                )}
                            >
                                {getGenderText(passenger.gender)}
                            </Meta>
                        )}
                    </div>
                )}
                {passenger.itn && (
                    <Meta
                        name={i18nBlock.itn()}
                        className={cx('metaItem')}
                        {...prepareQaAttributes('passenger-info-itn')}
                    >
                        {passenger.itn}
                    </Meta>
                )}
                {passenger.phone && (
                    <Meta
                        name={i18nBlock.phone()}
                        className={cx('metaItem')}
                        {...prepareQaAttributes('passenger-info-phone')}
                    >
                        {passenger.phone}
                    </Meta>
                )}
                {passenger.email && (
                    <Meta
                        name={i18nBlock.email()}
                        className={cx('metaItem', 'flexItemOverflow')}
                        {...prepareQaAttributes('passenger-info-email')}
                    >
                        {passenger.email}
                    </Meta>
                )}
            </div>
        );
    }

    private renderModal() {
        const {deviceType, passenger} = this.props;
        const {isModalOpened} = this.state;

        return (
            <AccountModal
                isVisible={isModalOpened}
                onClose={this.closeModal}
                allowFullScreen
            >
                <div className={cx('modalContent')} ref={this.refModal}>
                    <FinalForm
                        onSubmit={this.editItem}
                        validate={this.validate}
                    >
                        {({
                            handleSubmit,
                            submitFailed,
                            errors,
                            values,
                            form,
                        }) => (
                            <PassengerEdit
                                initialize={form.initialize}
                                handleSubmit={handleSubmit}
                                canShowErrors={submitFailed}
                                formErrors={errors}
                                formValues={values}
                                modalRef={this.refModal.current}
                                form={ACCOUNT_PASSENGER_INDEX_FORM}
                                isVisible={isModalOpened}
                                deviceType={deviceType}
                                onModalClose={this.closeModal}
                                data={passenger}
                                finalFormManager
                            />
                        )}
                    </FinalForm>
                </div>
            </AccountModal>
        );
    }

    render() {
        const {
            deviceType: {isMobile},
            fetching,
            passenger,
        } = this.props;

        return (
            <React.Fragment>
                {passenger && (
                    <div
                        className={cx('info', {
                            info_mobile: isMobile,
                        })}
                        {...prepareQaAttributes('passenger-details')}
                    >
                        {this.renderNick()}
                        {this.renderMeta()}
                    </div>
                )}
                {this.renderModal()}
                {isMobile && fetching && (
                    <Preloader
                        {...prepareQaAttributes('passenger-page-loading')}
                    />
                )}
            </React.Fragment>
        );
    }
}

export default withRouter(PassengerInfo);
