import React, {PureComponent, ReactNode} from 'react';
import _debounce from 'lodash/debounce';
import {Form as FinalForm} from 'react-final-form';

import {ACCOUNT_PASSENGER_FULL_FORM} from 'constants/reduxForm';
import {FORM_FIELDS} from 'constants/account';

import {IWithDeviceType} from 'types/withDeviceType';
import {EAccountGoal} from 'utilities/metrika/types/goals/account';
import {ICountry} from 'types/common/ICountry';
import {
    IAddFormValuesProps,
    IErrorsProps,
} from 'projects/account/pages/Passengers/types';

import {IPreparedPassenger} from 'reducers/account/passengers/api-types';
import {IDocTypesProps} from 'reducers/account/docTypes/api-types';

import {reachGoal} from 'utilities/metrika';
import {deviceModMobile} from 'utilities/stylesUtils';
import validateAndGetNewValues from 'projects/account/pages/Passengers/utilities/validateForm/passengerAddFormValidate';
import TDataAddPassenger from 'src/utilities/passengerApiMethods/TDataAddPassenger';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/account';

import SearchInput from 'projects/account/components/SearchInput/SearchInput';
import PassengerAdd from 'projects/account/pages/Passengers/components/PassengerAdd/PassengerAdd';
import AccountEmpty from 'projects/account/components/AccountEmpty/AccountEmpty';
import AccountModal from 'projects/account/components/AccountModal/AccountModal';

import PassengerListButtons from './blocks/PassengerListButtons/PassengerListButtons';
import PassengerListItem from './blocks/PassengerListItem/PassengerListItem';

import cx from './PassengerList.scss';

interface IPassengerListProps extends IWithDeviceType {
    addPassenger: (values: TDataAddPassenger) => void;
    items: IPreparedPassenger[];
    docTypes: IDocTypesProps;
    userId: string;
    citizenship: ICountry[];
}

interface IPassengerListState {
    isModalOpened: boolean;
    searchVal: string;
    searchItems: IPreparedPassenger[];
}

export default class PassengerList extends PureComponent<
    IPassengerListProps,
    IPassengerListState
> {
    state = {
        isModalOpened: false,
        searchVal: '',
        searchItems: [],
    };

    refModal = React.createRef<HTMLDivElement>();

    static getDerivedStateFromProps(
        nextProps: IPassengerListProps,
        prevState: IPassengerListState,
    ): IPassengerListState {
        if (!prevState.searchVal) {
            return {
                ...prevState,
                searchItems: nextProps.items,
            };
        }

        return prevState;
    }

    private handleFormSubmit = (formValues: IAddFormValuesProps): void => {
        const {docTypes} = this.props;

        const {values} = validateAndGetNewValues(formValues, docTypes);
        const selectedType = Array.isArray(values.type)
            ? values.type[0]
            : values.type;

        const rules = docTypes.rules[selectedType];
        const documentFields: TDataAddPassenger['document'] = {
            type: selectedType,
        };

        (Object.keys(values) as (keyof typeof values)[]).forEach(key => {
            const value = values[key];

            if (Array.isArray(value)) {
                const firstItem = value[0];

                (values[key] as typeof firstItem) = firstItem;
            }

            if (
                FORM_FIELDS[key].documents &&
                (!rules.unused.includes(key) || key === 'citizenship')
            ) {
                documentFields[key] = values[key];
            }
        });

        const title = [
            values.lastName || values.lastNameEn,
            values.firstName || values.firstNameEn,
        ]
            .join(' ')
            .trim();

        const result = {
            passenger: {
                title: title || i18nBlock.infoDotTitleDotDefaultValue(),
                birthDate: values.birthDate,
                gender: values.gender,
                itn: values.itn,
                email: values.email,
                phone: values.phone,
                train_notifications_enabled: values.train_notifications_enabled,
            },
            document: {
                ...documentFields,
            },
        };

        this.setState(
            {
                isModalOpened: false,
            },
            () => this.props.addPassenger(result),
        );
    };

    private validate = (formValues: IAddFormValuesProps): IErrorsProps => {
        const {docTypes} = this.props;
        const {errors} = validateAndGetNewValues(formValues, docTypes);

        return errors;
    };

    private openModal = (): void => {
        reachGoal(EAccountGoal.ADD_PASSENGER_BUTTON);

        this.setState({
            isModalOpened: true,
        });
    };

    private closeModal = (): void => {
        this.setState({
            isModalOpened: false,
        });
    };

    private onSearchChange = (text: string): void => {
        this.setState({
            searchVal: text,
        });

        _debounce(this.updateItems, 200)();
    };

    private updateItems = (): void => {
        const {items} = this.props;
        const {searchVal} = this.state;

        this.setState({
            searchItems: items.filter(
                item =>
                    item.title
                        .toLowerCase()
                        .indexOf(searchVal.toLowerCase()) !== -1,
            ),
        });
    };

    private renderList(): ReactNode {
        const {
            userId,
            items,
            deviceType,
            deviceType: {isMobile},
        } = this.props;
        const {searchVal, searchItems} = this.state;
        const imageSize = isMobile ? 160 : 210;

        if (items.length) {
            return (
                <div
                    className={cx('list', {
                        list_mobile: isMobile,
                    })}
                >
                    <SearchInput
                        className={cx(
                            'input',
                            deviceModMobile('input', deviceType),
                        )}
                        value={searchVal}
                        placeholder={i18nBlock.listDotSearchDotPlaceholder()}
                        onChange={this.onSearchChange}
                    />
                    <div className={cx('inner')}>
                        {searchItems.length === 0 && (
                            <div className={cx('notfound')}>
                                {i18nBlock.listDotNotFound()}
                            </div>
                        )}
                        {searchItems.map((item: IPreparedPassenger) => (
                            <PassengerListItem
                                key={item.id}
                                {...item}
                                isActive={item.id === userId}
                                isMobile={isMobile}
                                {...prepareQaAttributes({
                                    key: item.id,
                                    current: 'passenger-list-item',
                                })}
                            />
                        ))}
                    </div>
                    <PassengerListButtons onOpenAddModal={this.openModal} />
                </div>
            );
        }

        return (
            <AccountEmpty
                isMobile={isMobile}
                image={{
                    width: imageSize,
                    height: imageSize,
                }}
                title={i18nBlock.listDotEmptyDotTitle()}
                smallText={i18nBlock.listDotEmptyDotText()}
                button={{
                    onClick: this.openModal,
                    text: i18nBlock.listDotEmptyDotAdd(),
                }}
                {...prepareQaAttributes('empty-passenger-list')}
            />
        );
    }

    private renderModal(): ReactNode {
        const {deviceType, citizenship, docTypes} = this.props;
        const {isModalOpened} = this.state;

        return (
            <AccountModal
                isVisible={isModalOpened}
                onClose={this.closeModal}
                allowFullScreen
            >
                <div className={cx('modalContent')} ref={this.refModal}>
                    <FinalForm
                        onSubmit={this.handleFormSubmit}
                        validate={this.validate}
                    >
                        {({
                            handleSubmit,
                            submitFailed,
                            errors,
                            values,
                            form,
                        }): ReactNode => (
                            <PassengerAdd
                                handleSubmit={handleSubmit}
                                canShowErrors={submitFailed}
                                formErrors={errors}
                                formValues={values}
                                modalRef={this.refModal.current}
                                change={form.change}
                                form={ACCOUNT_PASSENGER_FULL_FORM}
                                citizenship={citizenship}
                                docTypes={docTypes}
                                deviceType={deviceType}
                                onModalClose={this.closeModal}
                                finalFormManager
                            />
                        )}
                    </FinalForm>
                </div>
            </AccountModal>
        );
    }

    render(): ReactNode {
        return (
            <React.Fragment>
                {this.renderList()}
                {this.renderModal()}
            </React.Fragment>
        );
    }
}
