import React, {PureComponent, ReactNode} from 'react';
import {withRouter, Link, RouteComponentProps} from 'react-router-dom';

import {IPreparedPassenger} from 'reducers/account/passengers/api-types';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import scrollTo from 'utilities/dom/scrollTo';
import {accountURLs} from 'projects/account/utilities/urls';

import ArrowRightIcon from 'icons/16/ArrowRight';

import cx from './PassengerListItem.scss';

interface IPassengerListItemProps
    extends IPreparedPassenger,
        RouteComponentProps,
        IWithQaAttributes {
    isActive?: boolean;
    isMobile: boolean;
    onCheck?: () => void;
}

class PassengerListItem extends PureComponent<IPassengerListItemProps> {
    private goToLink = () => {
        const {isMobile} = this.props;

        if (!isMobile) {
            scrollTo({top: 0, left: 0});
        }
    };

    private renderInfo(): ReactNode {
        return (
            <React.Fragment>
                {this.renderName()}
                {this.renderRight()}
            </React.Fragment>
        );
    }

    private renderName(): ReactNode {
        const {title, birthDate} = this.props;

        return (
            <div className={cx('left')}>
                <div
                    className={cx('passengerName')}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'title',
                    })}
                >
                    {title}
                </div>

                <div className={cx('birthdate')}>{birthDate}</div>
            </div>
        );
    }

    private renderRight(): ReactNode {
        const {isMobile} = this.props;

        if (isMobile) {
            return <ArrowRightIcon className={cx('icon')} />;
        }

        return null;
    }

    render(): ReactNode {
        const {id, isActive, isMobile} = this.props;

        return (
            <Link
                to={{
                    pathname: accountURLs.getPassengerUrl(id),
                    state: {scrollTop: null},
                }}
                className={cx('item', {
                    item_active: !isMobile && isActive,
                    item_mobile: isMobile,
                })}
                onClick={this.goToLink}
                {...prepareQaAttributes(this.props)}
            >
                {this.renderInfo()}
            </Link>
        );
    }
}

export default withRouter(PassengerListItem);
