import React, {PureComponent} from 'react';

import {EAccountGoal} from 'utilities/metrika/types/goals/account';

import {reachGoal} from 'utilities/metrika';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/account';

import Button from 'components/Button/Button';
import AccountModal from 'projects/account/components/AccountModal/AccountModal';
import RemoveDialog from 'projects/account/pages/Passengers/components/RemoveDialog/RemoveDialog';

import cx from './PassengerRemove.scss';

interface IPassengerRemoveProps {
    onRemoveItem: () => void;
    isMobile: boolean;
}

interface IPassengerRemoveState {
    isModalOpened: boolean;
}

export default class PassengerRemove extends PureComponent<
    IPassengerRemoveProps,
    IPassengerRemoveState
> {
    state = {
        isModalOpened: false,
    };

    private openModal = (): void => {
        reachGoal(EAccountGoal.REMOVE_PASSENGER_BUTTON);
        this.setState({
            isModalOpened: true,
        });
    };

    private closeModal = (): void =>
        this.setState({
            isModalOpened: false,
        });

    private handleSubmit = (): void => {
        this.closeModal();
        this.props.onRemoveItem();
    };

    private renderModal(): React.ReactNode {
        const {isMobile} = this.props;
        const {isModalOpened} = this.state;

        return (
            <AccountModal isVisible={isModalOpened} onClose={this.closeModal}>
                <RemoveDialog
                    isAbsolute={false}
                    isMobile={isMobile}
                    title={i18nBlock.removeDotTitle()}
                    text={i18nBlock.removeDotText()}
                    onClose={this.closeModal}
                    onSubmit={this.handleSubmit}
                    {...prepareQaAttributes('account-remove-confirmation')}
                />
            </AccountModal>
        );
    }

    private renderButton(): React.ReactNode {
        const {isMobile} = this.props;

        return (
            <div
                className={cx('delete', {
                    delete_mobile: isMobile,
                })}
            >
                <Button
                    theme="attention"
                    width={isMobile ? 'max' : 'auto'}
                    size={isMobile ? 'l' : 'm-inset'}
                    className={cx('removeButton')}
                    onClick={this.openModal}
                    {...prepareQaAttributes('remove-passenger')}
                >
                    {i18nBlock.removeDotButton()}
                </Button>
            </div>
        );
    }

    render(): React.ReactNode {
        return (
            <React.Fragment>
                {this.renderButton()}
                {this.renderModal()}
            </React.Fragment>
        );
    }
}
