import React, {PureComponent} from 'react';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/account';

import Button from 'components/Button/Button';

import cx from './RemoveDialog.scss';

interface IPassengerRemoveDialogProps extends IWithQaAttributes {
    isAbsolute: boolean;
    isMobile: boolean;
    title: string;
    text?: string;
    actionText: string;
    onClose: () => void;
    onSubmit: () => void;
}

export default class RemoveDialog extends PureComponent<IPassengerRemoveDialogProps> {
    static defaultProps = {
        isAbsolute: false,
        actionText: i18nBlock.removeDotDialogDotAction(),
    };

    private renderTitle(): React.ReactNode {
        const {title} = this.props;

        return <div className={cx('title')}>{title}</div>;
    }

    private renderText(): React.ReactNode {
        const {text} = this.props;

        if (text && text.length > 0) {
            return <div className={cx('text')}>{text}</div>;
        }

        return null;
    }

    private renderControls(): React.ReactNode {
        return (
            <div className={cx('controls')}>
                {this.renderAction()}
                {this.renderCancel()}
            </div>
        );
    }

    private renderAction(): React.ReactNode {
        const {isMobile, onSubmit, actionText} = this.props;

        return (
            <div
                className={cx('control')}
                {...prepareQaAttributes('account-remove-confirmation')}
            >
                <Button
                    theme="secondary"
                    size={isMobile ? 'l' : 'm-inset'}
                    width="max"
                    onClick={onSubmit}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'submit',
                    })}
                >
                    {actionText}
                </Button>
            </div>
        );
    }

    private renderCancel(): React.ReactNode {
        const {isMobile, onClose} = this.props;

        return (
            <div className={cx('control')}>
                <Button
                    theme="primary"
                    size={isMobile ? 'l' : 'm-inset'}
                    width="max"
                    onClick={onClose}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'cancel',
                    })}
                >
                    {i18nBlock.removeDotDialogDotCancel()}
                </Button>
            </div>
        );
    }

    render(): React.ReactNode {
        const {isMobile, isAbsolute} = this.props;

        return (
            <div
                className={cx('wrap', {
                    wrap_mobile: isMobile,
                    wrap_absolute: isAbsolute,
                })}
            >
                {this.renderTitle()}
                {this.renderText()}
                {this.renderControls()}
            </div>
        );
    }
}
