import {
    IAddFormValuesProps,
    IErrorsProps,
} from 'projects/account/pages/Passengers/types';

import {IDocTypesProps} from 'reducers/account/docTypes/api-types';

import {
    DateLikeType,
    isDateInFuture,
    isValidDateForBackend,
} from 'utilities/dateUtils/index';
import {isValidItn} from 'utilities/validation/isValidItn';
import {isRussianPhoneNumberStartsWithPlus} from 'projects/trains/lib/order/fields/phone';
import isImEmail from 'projects/account/pages/Passengers/utilities/validateForm/isImEmail';

import * as i18nBlock from 'i18n/accountForms';

import validateForm from './documentValidate';
import checkEnNames from './checkEnNames';

interface IValidationResultProps {
    errors: IErrorsProps;
    values: IAddFormValuesProps;
}

const validateBirthDate = (birthDate?: DateLikeType): string | undefined => {
    if (!birthDate) return i18nBlock.errorsDotRequiredDotBirthDate();

    if (!isValidDateForBackend(birthDate))
        return i18nBlock.errorsDotBirthDateDotWrong();

    if (isDateInFuture(birthDate))
        return i18nBlock.errorsDotBirthDateDotFuture();
};

const validateExpirationDate = (
    expirationDate?: DateLikeType,
): string | undefined => {
    if (!expirationDate) return;

    if (!isValidDateForBackend(expirationDate)) {
        return i18nBlock.errorsDotExpirationDateDotWrong();
    }

    if (!isDateInFuture(expirationDate)) {
        return i18nBlock.errorsDotExpirationDateDotPast();
    }
};

// Эта функция не только валидирует данные (несмотря на ее название), но возвращает новые значения:
// например, в случае, если в firstName будет введен текст на латинице, оно будет переименовано в firstNameEn
export default (
    vals: IAddFormValuesProps,
    docTypes: IDocTypesProps,
): IValidationResultProps => {
    let rules,
        values: IAddFormValuesProps = {...vals};

    let errors: IErrorsProps = {};

    const birthDateError = validateBirthDate(values.birthDate);

    if (birthDateError) {
        errors.birthDate = birthDateError;
    }

    if (!values.gender) {
        errors.gender = i18nBlock.errorsDotRequiredDotGender();
    }

    if (!values.citizenship) {
        errors.citizenship = i18nBlock.errorsDotRequiredDotCitizenship();
    }

    const expirationDateError = validateExpirationDate(values.expirationDate);

    if (expirationDateError) {
        errors.expirationDate = expirationDateError;
    }

    if (values.type) {
        // Проверяем есть ли тип в справочнике
        const key = Array.isArray(values.type) ? values.type[0] : values.type;

        if (docTypes.items.includes(key)) {
            rules = docTypes.rules[key];
            errors = {
                ...errors,
                ...validateForm(values, rules),
            };

            ({errors, values} = checkEnNames({values, rules, errors}));
        } else {
            errors.type = i18nBlock.errorsDotRequiredDotType();
        }
    } else {
        errors.type = i18nBlock.errorsDotRequiredDotType();
    }

    if (values.itn && !isValidItn(values.itn)) {
        errors.itn = i18nBlock.errorsDotItnDotWrong();
    }

    if (values.email && !isImEmail(values.email)) {
        errors.email = i18nBlock.errorsDotEmailDotWrong();
    }

    if (values.phone && !isRussianPhoneNumberStartsWithPlus(values.phone)) {
        errors.phone = i18nBlock.errorsDotPhoneDotWrong();
    }

    if (
        !(errors.email || errors.phone) &&
        values.train_notifications_enabled &&
        !values.email &&
        !values.phone
    ) {
        errors.email = errors.phone = i18nBlock.errorsDotAgreeDotWrong();
    }

    return {
        errors,
        values,
    };
};
