import {FC, memo, useEffect, useMemo} from 'react';
import {useParams} from 'react-router-dom';
import {useDispatch, useSelector} from 'react-redux';

import {EProjectName} from 'constants/common';

import {EFooterProject} from 'components/Footer/types';
import EHeaderBorderBottomType from 'components/Header/types/EHeaderBorderBottomType';
import {ETripsGoal} from 'utilities/metrika/types/goals/trips';

import {
    fetchTrip,
    restoreHiddenHotelCrossSale,
} from 'reducers/trips/tripPage/thunk';
import {CustomDispatch} from 'reducers/trains/customDispatch';
import {getTrip} from 'reducers/trips/tripPage/actions';

import {tripSelector} from 'selectors/trips/tripPage/tripSelector';

import useServerDataFetcher from 'utilities/hooks/useServerDataFetcher';
import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {reachGoal} from 'utilities/metrika';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {isHotelOrderBlock} from 'projects/account/pages/TripPage/utilities/blocks';

import LayoutDefault, {
    ELayoutBgColor,
} from 'components/Layouts/LayoutDefault/LayoutDefault';
import HangingYandexMessenger from 'components/YandexMessenger/components/HangingYandexMessenger/HangingYandexMessenger';
import TripContent from 'projects/account/pages/TripPage/components/TripContent/TripContent';

import {usePlatform} from 'contexts/PlatformContext';

import cx from './TripPage.scss';

const ROOT_QA = 'tripPage';

const TripPage: FC = () => {
    const {tripId} = useParams<{tripId: string}>();
    const dispatch: CustomDispatch = useDispatch();
    const deviceType = useDeviceType();
    const {isWeb} = usePlatform();
    const trip = useSelector(tripSelector);

    const {isDesktop} = deviceType;

    const hasHotelOrders = useMemo(() => {
        if (!trip.value) {
            return false;
        }

        return trip.value.blocks.some(
            block => isHotelOrderBlock(block) && block.orders.length > 0,
        );
    }, [trip.value]);

    useServerDataFetcher([
        (): void => {
            dispatch(getTrip.request({tripId}));
        },
    ]);

    useEffect(() => {
        dispatch(fetchTrip({tripId}));
        dispatch(restoreHiddenHotelCrossSale());

        reachGoal(ETripsGoal.TRIP_PAGE_SHOW);
    }, [dispatch, tripId]);

    return (
        <LayoutDefault
            className={cx('root', deviceMods('root', deviceType))}
            bgColor={
                deviceType.isMobile ? ELayoutBgColor.DARK : ELayoutBgColor.LIGHT
            }
            showNavigation={isDesktop}
            searchFormInitialIsExpanded={false}
            project={EProjectName.ACCOUNT}
            footerType={EFooterProject.TRIPS}
            headerBorderBottomType={EHeaderBorderBottomType.NONE}
            footerIncludeSupportBlock
            footerClassName={cx('footer')}
            contentClassName={cx('content')}
            showHeader={isWeb}
            showFooter={isWeb}
            {...prepareQaAttributes(ROOT_QA)}
        >
            <TripContent {...prepareQaAttributes(ROOT_QA)} />

            {trip.value && !hasHotelOrders && (
                <HangingYandexMessenger
                    entrypoint="tripPage"
                    metrikaGoal={ETripsGoal.TRIP_PAGE_CHAT_CLICK}
                />
            )}
        </LayoutDefault>
    );
};

export default memo(TripPage);
