import React, {useCallback, useEffect, useMemo} from 'react';
import {InView} from 'react-intersection-observer';

import {IWithClassName} from 'types/withClassName';
import TTripActivitiesBlock from 'types/trips/TTripBlock/TTripActivitiesBlock';
import {ETripsGoal} from 'utilities/metrika/types/goals/trips';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import prepareActivityProps from 'projects/account/pages/TripPage/components/ActivitiesBlock/utilities/prepareActivityProps';
import {reachGoal} from 'utilities/metrika';
import {useReachGoal} from 'utilities/metrika/useReachGoal';
import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/account-TripPage';

import Heading from 'components/Heading/Heading';
import Flex from 'components/Flex/Flex';
import Activity from 'projects/account/pages/TripPage/components/ActivitiesBlock/components/Activity/Activity';
import ActivityTypeFilter from './components/ActivityTypeFilter/ActivityTypeFilter';
import useActivityTypeFilter from 'projects/account/pages/TripPage/components/ActivitiesBlock/hooks/useActivityTypeFilter';

import MobileCard from '../MobileCard/MobileCard';

import cx from './ActivitiesBlock.scss';

interface IActivitiesBlockProps extends IWithClassName, IWithQaAttributes {
    activitiesBlock: TTripActivitiesBlock;
}

const ActivitiesBlock: React.FC<IActivitiesBlockProps> = props => {
    const {className, activitiesBlock} = props;

    const deviceType = useDeviceType();

    const rootQa = getQa(props);

    const activities = useMemo(() => {
        return activitiesBlock.isLoaded ? activitiesBlock.block.activities : [];
    }, [activitiesBlock]);

    const handleActivityClick = useReachGoal(ETripsGoal.ACTIVITY_CLICK);

    const handleChangeInView = useCallback((inView: boolean): void => {
        if (!inView) {
            return;
        }

        reachGoal(ETripsGoal.ACTIVITIES_SCROLL);
    }, []);

    const activityTypeFilter = useActivityTypeFilter(activities);

    const content = useMemo(() => {
        if (!activityTypeFilter.filtered.length) {
            return null;
        }

        return (
            <InView
                as="div"
                threshold={0.1}
                triggerOnce
                onChange={handleChangeInView}
                {...prepareQaAttributes(rootQa)}
            >
                <Flex flexDirection="column" between={5}>
                    <Flex flexDirection="column" between={3}>
                        <Heading
                            level={2}
                            {...prepareQaAttributes({
                                parent: rootQa,
                                current: 'title',
                            })}
                        >
                            {i18nBlock.activitiesTitle()}
                        </Heading>

                        <ActivityTypeFilter
                            available={activityTypeFilter.available}
                            active={activityTypeFilter.active}
                            onChange={activityTypeFilter.setActive}
                            {...prepareQaAttributes({
                                parent: rootQa,
                                current: 'activityTypeFilter',
                            })}
                        />
                    </Flex>

                    <div className={cx('activities')}>
                        {activityTypeFilter.filtered.map((activity, index) => {
                            const activityProps =
                                prepareActivityProps(activity);

                            if (!activityProps) {
                                return null;
                            }

                            return (
                                <Activity
                                    key={index}
                                    className={cx('activity')}
                                    onClick={handleActivityClick}
                                    {...activityProps}
                                    {...prepareQaAttributes({
                                        parent: rootQa,
                                        current: 'activity',
                                        key: index,
                                    })}
                                />
                            );
                        })}
                    </div>
                </Flex>
            </InView>
        );
    }, [handleActivityClick, handleChangeInView, activityTypeFilter, rootQa]);

    const hasContent = Boolean(content);

    useEffect(() => {
        if (!hasContent) {
            return;
        }

        reachGoal(ETripsGoal.ACTIVITIES_SHOW);
    }, [hasContent]);

    if (!hasContent) {
        return null;
    }

    const rootClassName = cx('root', deviceMods('root', deviceType), className);

    if (deviceType.isMobile) {
        return (
            <MobileCard className={rootClassName} y={5}>
                {content}
            </MobileCard>
        );
    }

    return <div className={rootClassName}>{content}</div>;
};

export default React.memo(ActivitiesBlock);
