import React, {useCallback} from 'react';

import {EAppActions} from 'constants/platforms/TPlatforms';

import ETripImageType from 'projects/trips/components/TripImage/types/ETripImageType';
import {IWithClassName} from 'types/withClassName';

import IPrice from 'utilities/currency/PriceInterface';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import TripImage from 'projects/trips/components/TripImage/TripImage';
import Text from 'components/Text/Text';
import DotSeparator from 'components/DotSeparator/DotSeparator';
import Intersperse from 'components/Intersperse/Intersperse';
import Price from 'components/Price/Price';
import Flex from 'components/Flex/Flex';
import Link from 'components/Link/Link';
import Label, {ECornersType, ELabelThemeType} from 'components/Label/Label';

import {useCoordinator} from 'contexts/PlatformContext';

import cx from './Activity.scss';

export interface IActivityProps extends IWithClassName, IWithQaAttributes {
    imageSrc: string;
    title: string;
    description: string;
    label: string;
    price?: IPrice | null;
    date?: string | null;
    url?: string;
    onClick?(): void;
}

const Activity: React.FC<IActivityProps> = props => {
    const {
        className,
        imageSrc,
        title,
        description,
        label,
        price,
        date,
        url,
        onClick,
    } = props;

    const coordinator = useCoordinator();

    const handleLinkClick = useCallback<React.MouseEventHandler>(
        event => {
            const processed = coordinator.doAction(
                EAppActions.NAVIGATE_TO_EXTERNAL_URL,
                url,
            );

            if (processed) {
                event.preventDefault();
            }
        },
        [url, coordinator],
    );

    return (
        <div
            className={cx('root', className)}
            onClick={onClick}
            {...prepareQaAttributes(props)}
        >
            <Link
                className={cx('imageLink')}
                url={url}
                target="_blank"
                rel="noopener noreferrer"
                onClick={handleLinkClick}
            >
                <TripImage
                    type={ETripImageType.AUTO_HEIGHT}
                    src={imageSrc}
                    useDefaultTravelImageStub
                    enableZoom
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'image',
                    })}
                />
            </Link>
            <Label
                className={cx('label')}
                size="m"
                theme={ELabelThemeType.NEUTRAL}
                cornersType={ECornersType.ROUNDED}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'label',
                })}
            >
                {label}
            </Label>
            <Flex flexDirection="column" above={2}>
                <Text
                    size="l"
                    weight="bold"
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'title',
                    })}
                >
                    {title}
                </Text>
                {(description || price) && (
                    <Text size="m">
                        <Intersperse separator={<DotSeparator />}>
                            {description}
                            {price && <Price {...price} />}
                        </Intersperse>
                    </Text>
                )}
                {date && (
                    <Text color="secondary" size="m">
                        {date}
                    </Text>
                )}
            </Flex>
            {url && (
                <Link
                    className={cx('link')}
                    url={url}
                    target="_blank"
                    rel="noopener noreferrer"
                    onClick={handleLinkClick}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'link',
                    })}
                />
            )}
        </div>
    );
};

export default Activity;
