import React, {useMemo} from 'react';

import ETripActivityType from 'types/trips/ITripActivities/ETripActivityType';
import {IWithClassName} from 'types/withClassName';

import getFilterItems from './utilities/getFilterItems';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Flex from 'components/Flex/Flex';
import CheckButton from 'components/CheckButton/CheckButton';
import HorizontalScroller from 'components/HorizontalScroller/HorizontalScroller';

interface IActivityTypeFilterProps extends IWithClassName, IWithQaAttributes {
    available: ETripActivityType[];
    active: ETripActivityType | null;
    onChange(active: ETripActivityType | null): void;
}

const ActivityTypeFilter: React.FC<IActivityTypeFilterProps> = props => {
    const {className, available, active, onChange} = props;

    const deviceType = useDeviceType();

    const items = useMemo(() => getFilterItems(available), [available]);

    if (!items.length) {
        return null;
    }

    const rootQa = getQa(props);

    const content = (
        <Flex inline nowrap between={2}>
            {items.map(item => (
                <CheckButton
                    key={item.key}
                    size="m"
                    checked={active === item.value}
                    onClick={(): void => {
                        onChange(item.value);
                    }}
                    {...prepareQaAttributes({
                        parent: rootQa,
                        current: 'activityType',
                        key: item.key,
                    })}
                >
                    {item.text}
                </CheckButton>
            ))}
        </Flex>
    );

    if (deviceType.isMobile) {
        return (
            <HorizontalScroller
                className={className}
                {...prepareQaAttributes(rootQa)}
            >
                {content}
            </HorizontalScroller>
        );
    }

    return (
        <div className={className} {...prepareQaAttributes(rootQa)}>
            {content}
        </div>
    );
};

export default ActivityTypeFilter;
