import React, {useCallback} from 'react';

import ITripAviaOrder from 'types/trips/ITripAviaOrder';
import {IWithClassName} from 'types/withClassName';
import {ETripsGoal} from 'utilities/metrika/types/goals/trips';

import {useReachGoal} from 'utilities/metrika/useReachGoal';
import {
    IWithQaAttributes,
    prepareQaAttributes,
    TQaValue,
} from 'utilities/qaAttributes/qaAttributes';
import getOrderIsCancelled from 'projects/account/pages/TripPage/utilities/getOrderIsCancelled';

import * as i18nBlock from 'i18n/account-TripPage';

import OrdersBlock from 'projects/account/pages/TripPage/components/OrdersBlock/OrdersBlock';
import Logo from 'projects/account/pages/TripPage/components/AirlinesLogo/AirlinesLogo';
import OrderMainInfo from 'projects/account/pages/TripPage/components/OrderMainInfo/OrderMainInfo';
import PNR from 'projects/account/pages/TripPage/components/AviaOrdersBlock/components/PNR/PNR';

interface IAviaOrdersBlockProps extends IWithClassName, IWithQaAttributes {
    orders: ITripAviaOrder[];
}

const AviaOrdersBlock: React.FC<IAviaOrdersBlockProps> = props => {
    const {className, orders} = props;

    const onOrderNavigate = useReachGoal(ETripsGoal.AVIA_ORDER_CLICK);

    const renderImage = useCallback(
        (order: ITripAviaOrder, orderQa?: TQaValue) => {
            const isCancelled = getOrderIsCancelled(order);

            return (
                <Logo
                    isCancelled={isCancelled}
                    airlines={order.airlines}
                    size="small"
                    {...prepareQaAttributes({
                        parent: orderQa,
                        current: 'logo',
                    })}
                />
            );
        },
        [],
    );

    const renderOrderMainInfo = useCallback(
        (order: ITripAviaOrder, orderQa?: TQaValue) => {
            const isCancelled = getOrderIsCancelled(order);

            return (
                <OrderMainInfo
                    isCancelled={isCancelled}
                    {...order}
                    {...prepareQaAttributes({
                        parent: orderQa,
                        current: 'orderMainInfo',
                    })}
                />
            );
        },
        [],
    );

    const renderOrderBottom = useCallback(
        (order: ITripAviaOrder, orderQa?: TQaValue) => {
            return (
                <PNR
                    airlines={order.airlines}
                    pnr={order.pnr}
                    registrationUrl={order.registrationUrl}
                    {...prepareQaAttributes({
                        parent: orderQa,
                        current: 'pnr',
                    })}
                />
            );
        },
        [],
    );

    return (
        <OrdersBlock
            className={className}
            orders={orders}
            title={i18nBlock.aviaTickets({count: orders.length})}
            renderImage={renderImage}
            renderOrderMainInfo={renderOrderMainInfo}
            renderOrderBottom={renderOrderBottom}
            onOrderNavigate={onOrderNavigate}
            {...prepareQaAttributes(props)}
        />
    );
};

export default React.memo(AviaOrdersBlock);
