import React, {useCallback} from 'react';

import {IAirline} from 'types/trips/ITripAviaOrder';
import {ETripsGoal} from 'utilities/metrika/types/goals/trips';
import {IWithClassName} from 'types/withClassName';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {useBoolean} from 'utilities/hooks/useBoolean';
import copyToClipboard from 'utilities/copyToClipboard';
import {reachGoal} from 'utilities/metrika';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/account-TripPage';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Button from 'components/Button/Button';
import CopyIcon from 'icons/16/Copy';
import CopyPNRModal from 'projects/account/pages/TripPage/components/CopyPNRModal/CopyPNRModal';

import cx from './PNR.scss';

interface ICopyPNRModalProps extends IWithClassName, IWithQaAttributes {
    pnr: string;
    registrationUrl: string | null;
    airlines: IAirline[];
}

const PNR: React.FC<ICopyPNRModalProps> = props => {
    const {pnr, airlines, registrationUrl, className} = props;

    const deviceType = useDeviceType();
    const {
        setTrue: openCopyModal,
        setFalse: closeCopyModal,
        value: copyModalIsVisible,
    } = useBoolean(false);

    const handleCopyButtonClick = useCallback(async () => {
        reachGoal(ETripsGoal.AVIA_ORDER_COPY_PNR_BUTTON_CLICK);

        const success = await copyToClipboard({text: pnr});

        if (!success) {
            return;
        }

        openCopyModal();
    }, [openCopyModal, pnr]);

    const {isMobile} = deviceType;

    return (
        <Flex
            className={cx('root', deviceMods('root', deviceType), className)}
            alignItems="center"
            justifyContent="space-between"
            inline
            {...prepareQaAttributes(props)}
        >
            <Text
                className={cx('pnrDescription')}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'description',
                })}
            >
                {i18nBlock.pnr()}
            </Text>

            <div
                className={cx('pnr')}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'pnr',
                })}
            >
                <Text size="l">{pnr}</Text>
            </div>

            {registrationUrl && (
                <>
                    <Button
                        className={cx('copyButton')}
                        shape={isMobile ? 'circle' : undefined}
                        size="l"
                        onClick={handleCopyButtonClick}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'copyButton',
                        })}
                    >
                        {isMobile ? (
                            <CopyIcon />
                        ) : (
                            <Flex inline alignItems="center" between={2}>
                                <CopyIcon className={cx('copyIcon')} />

                                <Text>{i18nBlock.copy()}</Text>
                            </Flex>
                        )}
                    </Button>

                    <CopyPNRModal
                        pnr={pnr}
                        registrationUrl={registrationUrl}
                        airlines={airlines}
                        isVisible={copyModalIsVisible}
                        onClose={closeCopyModal}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'copyModal',
                        })}
                    />
                </>
            )}
        </Flex>
    );
};

export default React.memo(PNR);
