import React, {useCallback} from 'react';

import {IWithClassName} from 'types/withClassName';
import ITripBusOrder from 'types/trips/ITripBusOrder';
import {ETripsGoal} from 'utilities/metrika/types/goals/trips';

import {useReachGoal} from 'utilities/metrika/useReachGoal';
import {
    IWithQaAttributes,
    prepareQaAttributes,
    TQaValue,
} from 'utilities/qaAttributes/qaAttributes';
import getOrderIsCancelled from 'projects/account/pages/TripPage/utilities/getOrderIsCancelled';

import * as i18nBlock from 'i18n/account-TripPage';

import OrdersBlock from 'projects/account/pages/TripPage/components/OrdersBlock/OrdersBlock';
import Logo from 'projects/account/pages/TripPage/components/BusOrdersBlock/components/Logo/Logo';
import OrderMainInfo from 'projects/account/pages/TripPage/components/OrderMainInfo/OrderMainInfo';
import DescriptionAndActions from 'projects/account/pages/TripPage/components/BusOrdersBlock/components/DescriptionAndActions/DescriptionAndActions';

interface IBusOrdersBlockProps extends IWithClassName, IWithQaAttributes {
    orders: ITripBusOrder[];
}

const BusOrdersBlock: React.FC<IBusOrdersBlockProps> = props => {
    const {className, orders} = props;

    const onOrderNavigate = useReachGoal(ETripsGoal.BUS_ORDER_CLICK);

    const renderImage = useCallback((order: ITripBusOrder) => {
        const isCancelled = getOrderIsCancelled(order);

        return <Logo isCancelled={isCancelled} />;
    }, []);

    const renderOrderMainInfo = useCallback(
        (order: ITripBusOrder, orderQa?: TQaValue) => {
            const isCancelled = getOrderIsCancelled(order);

            return (
                <OrderMainInfo
                    isCancelled={isCancelled}
                    {...order}
                    {...prepareQaAttributes({
                        parent: orderQa,
                        current: 'orderMainInfo',
                    })}
                />
            );
        },
        [],
    );

    const renderOrderBottom = useCallback(
        (order: ITripBusOrder, orderQa?: TQaValue) => {
            return (
                <DescriptionAndActions
                    order={order}
                    {...prepareQaAttributes({
                        parent: orderQa,
                        current: 'descriptionAndActions',
                    })}
                />
            );
        },
        [],
    );

    return (
        <OrdersBlock
            className={className}
            orders={orders}
            title={i18nBlock.busTickets()}
            renderImage={renderImage}
            renderOrderMainInfo={renderOrderMainInfo}
            renderOrderBottom={renderOrderBottom}
            onOrderNavigate={onOrderNavigate}
            {...prepareQaAttributes(props)}
        />
    );
};

export default React.memo(BusOrdersBlock);
