import {FC, memo} from 'react';

import ITripBusOrder from 'types/trips/ITripBusOrder';

import getDownloadFilename from 'projects/account/utilities/downloadBlank/getDownloadFilename';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {usePrint} from 'utilities/hooks/usePrint';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDownloadBlankUrl} from 'projects/account/pages/TripPage/components/BusOrdersBlock/utilities/useDownloadBlankUrl';

import * as i18nBlock from 'i18n/account-TripPage';

import Flex from 'components/Flex/Flex';
import DownloadIcon from 'icons/16/Download';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import Button from 'components/Button/Button';
import PrintIcon from 'icons/16/Print';
import ButtonLink from 'components/ButtonLink/ButtonLink';
import Text from 'components/Text/Text';

import cx from './DescriptionAndActions.scss';

interface IDescriptionAndActionsProps extends IWithQaAttributes {
    order: ITripBusOrder;
}

const DescriptionAndActions: FC<IDescriptionAndActionsProps> = props => {
    const {order} = props;

    const {isMobile, isDesktop} = useDeviceType();
    const downloadUrl = useDownloadBlankUrl(order.id, order.downloadBlankToken);

    const {isPrinting, startPrinting} = usePrint({
        printable: downloadUrl,
        type: 'pdf',
    });

    return (
        <Flex
            className={cx('root')}
            inline
            flexWrap="nowrap"
            {...prepareQaAttributes(props)}
        >
            <Flex
                className={cx('descriptionAndSupplier')}
                flexDirection="column"
            >
                <Text
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'busDescription',
                    })}
                >
                    {order.description}
                </Text>

                {order.carrierName && (
                    <Text
                        color="secondary"
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'carrierName',
                        })}
                    >
                        {order.carrierName}
                    </Text>
                )}
            </Flex>

            <ButtonLink
                className={cx('downloadButton')}
                shape={isMobile ? 'circle' : undefined}
                size="l"
                disabled={!downloadUrl}
                url={downloadUrl}
                download={getDownloadFilename('order', order.id, 'pdf')}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'downloadButton',
                })}
            >
                {isMobile ? (
                    <DownloadIcon />
                ) : (
                    <TextWithIcon
                        text={i18nBlock.download()}
                        iconLeft={DownloadIcon}
                        iconLeftClassName={cx('downloadIcon')}
                    />
                )}
            </ButtonLink>

            {isDesktop && (
                <Button
                    className={cx('printButton')}
                    size="l"
                    icon={<PrintIcon />}
                    disabled={!downloadUrl || isPrinting}
                    onClick={startPrinting}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'printButton',
                    })}
                />
            )}
        </Flex>
    );
};

export default memo(DescriptionAndActions);
