import React, {useCallback} from 'react';

import {IWithClassName} from 'types/withClassName';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import copyToClipboard from 'utilities/copyToClipboard';

import * as i18nBlock from 'i18n/account-TripPage';

import CheckCircleIcon from 'icons/24/CheckCircle';
import CopyIcon from 'icons/24/Copy';
import Text from 'components/Text/Text';
import Separator from 'components/Separator/Separator';
import ButtonLink from 'components/ButtonLink/ButtonLink';
import Intersperse from 'components/Intersperse/Intersperse';
import Button from 'components/Button/Button';

import cx from './CopyPNRButtonMobile.scss';

interface ICopyPNRButtonMobileProps extends IWithClassName, IWithQaAttributes {
    copyText: string;
    description?: React.ReactNode;
    registrationUrl: string;
    isCopied: boolean;
    onClickCopyButton?(success: boolean): void;
}

const CopyPNRButtonMobile: React.FC<ICopyPNRButtonMobileProps> = props => {
    const {
        className,
        copyText,
        description = null,
        registrationUrl,
        isCopied,
        onClickCopyButton,
    } = props;

    const handleCopyButtonClick = useCallback(async () => {
        const success = await copyToClipboard({text: copyText});

        onClickCopyButton?.(success);
    }, [copyText, onClickCopyButton]);

    return (
        <div className={className}>
            <Intersperse separator={<Separator className={cx('separator')} />}>
                <Button
                    size="l"
                    theme="primary"
                    width="max"
                    iconLeft={isCopied ? <CheckCircleIcon /> : <CopyIcon />}
                    onClick={handleCopyButtonClick}
                    disabled={isCopied}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'copyButton',
                    })}
                >
                    {isCopied ? i18nBlock.pnrCopied() : i18nBlock.copy()}
                </Button>
                {isCopied && registrationUrl && (
                    <div>
                        {description && (
                            <Text
                                className={cx('description')}
                                size="m"
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'description',
                                })}
                            >
                                {description}
                            </Text>
                        )}
                        <ButtonLink
                            className={cx('actionButton')}
                            theme="primary"
                            size="l"
                            width="max"
                            target="_blank"
                            rel="noopener noreferrer"
                            url={registrationUrl}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'actionButton',
                            })}
                        >
                            {i18nBlock.copyModalPnrActionButton()}
                        </ButtonLink>
                    </div>
                )}
            </Intersperse>
        </div>
    );
};

export default CopyPNRButtonMobile;
