import React from 'react';

import {IAirline} from 'types/trips/ITripAviaOrder';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/account-TripPage';

import ModalOrBottomSheet from 'components/ModalOrBottomSheet/ModalOrBottomSheet';
import Logo from 'projects/account/pages/TripPage/components/AirlinesLogo/AirlinesLogo';
import Flex from 'components/Flex/Flex';
import Heading from 'components/Heading/Heading';
import Text from 'components/Text/Text';
import ButtonLink from 'components/ButtonLink/ButtonLink';
import CopyPNRButtonMobile from 'projects/account/pages/TripPage/components/CopyPNRButtonMobile/CopyPNRButtonMobile';

import cx from './CopyPNRModal.scss';

interface ICopyModalProps extends IWithQaAttributes {
    pnr: string;
    registrationUrl: string;
    airlines: IAirline[];
    isVisible: boolean;
    onClose(): void;
}

const CopyPNRModal: React.FC<ICopyModalProps> = props => {
    const {pnr, registrationUrl, airlines, isVisible, onClose} = props;

    const deviceType = useDeviceType();

    return (
        <ModalOrBottomSheet isVisible={isVisible} onClose={onClose}>
            <Flex
                className={cx('root', deviceMods('root', deviceType))}
                flexDirection="column"
                alignItems="center"
                {...prepareQaAttributes({
                    parent: props,
                    current: 'modalContent',
                })}
            >
                <Logo className={cx('logo')} airlines={airlines} size="big" />
                <Heading
                    className={cx('title')}
                    level={2}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'title',
                    })}
                >
                    {i18nBlock.copyModalPnrTitle2({
                        isMobile: deviceType.isMobile,
                    })}
                </Heading>
                <Text
                    className={cx('pnr')}
                    size="xxl"
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'pnr',
                    })}
                >
                    {pnr}
                </Text>
                {deviceType.isMobile ? (
                    <CopyPNRButtonMobile
                        className={cx('copyPNRButtonMobile')}
                        copyText={pnr}
                        description={i18nBlock.copyModalPnrDescription()}
                        registrationUrl={registrationUrl}
                        isCopied
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'copyPNRButtonMobile',
                        })}
                    />
                ) : (
                    <>
                        <Text
                            className={cx('description')}
                            size="m"
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'description',
                            })}
                        >
                            {i18nBlock.copyModalPnrDescription()}
                        </Text>
                        <ButtonLink
                            className={cx('actionButton')}
                            theme="primary"
                            size="m-inset"
                            target="_blank"
                            rel="noopener noreferrer"
                            url={registrationUrl}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'actionButton',
                            })}
                        >
                            {i18nBlock.copyModalPnrActionButton()}
                        </ButtonLink>
                    </>
                )}
            </Flex>
        </ModalOrBottomSheet>
    );
};

export default CopyPNRModal;
