import React, {useEffect} from 'react';

import {IAirline} from 'types/trips/ITripAviaOrder';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useBoolean} from 'utilities/hooks/useBoolean';

import * as i18nBlock from 'i18n/account-TripPage';

import ModalOrBottomSheet from 'components/ModalOrBottomSheet/ModalOrBottomSheet';
import Logo from 'projects/account/pages/TripPage/components/AirlinesLogo/AirlinesLogo';
import Flex from 'components/Flex/Flex';
import Heading from 'components/Heading/Heading';
import Text from 'components/Text/Text';
import ButtonLink from 'components/ButtonLink/ButtonLink';
import CopyPNRButtonMobile from 'projects/account/pages/TripPage/components/CopyPNRButtonMobile/CopyPNRButtonMobile';
import InputWithCopyButton from 'components/InputWithCopyButton/InputWithCopyButton';

import cx from './CopyPNRModalWithCopy.scss';

interface ICopyModalProps extends IWithQaAttributes {
    pnr: string;
    registrationUrl: string;
    airlines: IAirline[];
    isVisible: boolean;
    onClose(): void;
}

const CopyPNRModalWithCopy: React.FC<ICopyModalProps> = props => {
    const {pnr, registrationUrl, airlines, isVisible, onClose} = props;

    const deviceType = useDeviceType();

    const {
        value: isCopied,
        setTrue: setTrueIsCopied,
        setFalse: setFalseIsCopied,
    } = useBoolean(false);

    useEffect(() => {
        if (!isVisible) {
            setFalseIsCopied();
        }
    }, [isVisible, setFalseIsCopied]);

    const description = (
        <Text
            className={cx('description')}
            size="m"
            {...prepareQaAttributes({
                parent: props,
                current: 'description',
            })}
        >
            {i18nBlock.copyModalPnrDescriptionCanCopy()}
        </Text>
    );

    return (
        <ModalOrBottomSheet isVisible={isVisible} onClose={onClose}>
            <Flex
                className={cx('root', deviceMods('root', deviceType))}
                flexDirection="column"
                alignItems="center"
                {...prepareQaAttributes({
                    parent: props,
                    current: 'modalContent',
                })}
            >
                <Logo className={cx('logo')} airlines={airlines} size="big" />
                <Heading
                    className={cx('title')}
                    level={2}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'title',
                    })}
                >
                    {i18nBlock.copyModalPnrTitleCanCopy()}
                </Heading>
                {description}
                {deviceType.isMobile ? (
                    <>
                        <Text
                            className={cx('pnr')}
                            size="xxl"
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'pnr',
                            })}
                        >
                            {pnr}
                        </Text>
                        <CopyPNRButtonMobile
                            className={cx('copyPNRButtonMobile')}
                            copyText={pnr}
                            registrationUrl={registrationUrl}
                            isCopied={isCopied}
                            onClickCopyButton={setTrueIsCopied}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'copyPNRButtonMobile',
                            })}
                        />
                    </>
                ) : (
                    <>
                        <InputWithCopyButton
                            className={cx('inputWithCopyButton')}
                            readOnly
                            value={pnr}
                            isCopied={isCopied}
                            onIsCopiedChange={setTrueIsCopied}
                        />
                        <ButtonLink
                            className={cx('actionButton')}
                            size="m-inset"
                            target="_blank"
                            rel="noopener noreferrer"
                            url={registrationUrl}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'actionButton',
                            })}
                        >
                            {i18nBlock.copyModalPnrActionButtonCanCopy()}
                        </ButtonLink>
                    </>
                )}
            </Flex>
        </ModalOrBottomSheet>
    );
};

export default CopyPNRModalWithCopy;
