import React, {useCallback, useEffect, useMemo} from 'react';
import {useDispatch} from 'react-redux';

import ITripHotelCrossSale from 'types/trips/ITripHotelCrossSale';
import {EYtpReferer} from 'types/common/IAttribution';
import {ETripsGoal} from 'utilities/metrika/types/goals/trips';
import {IWithClassName} from 'types/withClassName';

import {hideHotelCrossSale} from 'reducers/trips/tripPage/thunk';

import {ROBOT} from 'utilities/dateUtils/formats';
import {parseDate} from 'utilities/dateUtils';
import {getCrossSaleHotelsSearchUrl} from 'projects/avia/utilities/getCrossSaleHotelsSearchUrl';
import getFormattedDatesForPeriod from 'projects/avia/utilities/getFormattedDatesForPeriod';
import {reachGoal} from 'utilities/metrika';
import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {isV3Response} from 'utilities/crossSaleHotels/isV3Response';
import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';

import * as i18nBlock from 'i18n/account-TripPage';

import useHotelsCrossSale from 'hooks/useHotelsCrossSale/useHotelsCrossSale';

import HotelsCrossSaleMap from 'components/HotelsCrossSaleMap/HotelsCrossSaleMap';
import Skeleton from 'projects/account/pages/TripPage/components/HotelCrossSaleBlock/components/Skeleton/Skeleton';
import Heading from 'components/Heading/Heading';
import Text from 'components/Text/Text';
import Flex from 'components/Flex/Flex';
import CloseIcon from 'icons/24/Close';
import MobileCard from 'projects/account/pages/TripPage/components/MobileCard/MobileCard';
import {useCoordinatorHotelCrossSaleMapClickCallback} from 'projects/account/pages/TripPage/components/HotelCrossSaleBlock/hooks/useCoordinatorHotelCrossSaleMapClickCallback';

import cx from './HotelCrossSaleBlock.scss';

interface IHotelCrossSaleBlockProps extends IWithQaAttributes, IWithClassName {
    crossSale: ITripHotelCrossSale;
    tripId: string;
}

const HotelCrossSaleBlock: React.FC<IHotelCrossSaleBlockProps> = props => {
    const {
        className,
        crossSale: {title, request},
        tripId,
    } = props;

    const dispatch = useDispatch();
    const deviceType = useDeviceType();

    const rootQa = getQa(props);
    const RootComponent = deviceType.isMobile ? MobileCard : 'div';

    const checkinDate = useMemo(() => {
        return request.checkinDate && parseDate(request.checkinDate);
    }, [request.checkinDate]);

    const checkoutDate = useMemo(() => {
        return request.checkoutDate && parseDate(request.checkoutDate);
    }, [request.checkoutDate]);

    const handleHide = useCallback(() => {
        dispatch(hideHotelCrossSale(tripId));

        reachGoal(ETripsGoal.HOTELS_CROSS_SALE_HIDE);
    }, [dispatch, tripId]);

    const {fetch, reset, isError, isLoading, data} = useHotelsCrossSale();

    const handleCrossSaleClick = useCoordinatorHotelCrossSaleMapClickCallback(
        request,
        data,
        checkinDate,
        checkoutDate,
    );

    const content = useMemo(() => {
        if (isError || !checkinDate || !checkoutDate) {
            return null;
        }

        const crossSale = data && (isV3Response(data) ? data.crossSale : data);

        if (isLoading) {
            return <Skeleton />;
        }

        if (!crossSale) {
            return null;
        }

        const {region, totalHotelCount} = crossSale;
        const searchUrl = getCrossSaleHotelsSearchUrl({
            geoId: region.geoId,
            offerSearchParams: {
                checkinDate: checkinDate.format(ROBOT),
                checkoutDate: checkoutDate.format(ROBOT),
                adults: request.adults,
                childrenAges: request.childrenAges,
            },
            ytpReferer: EYtpReferer.TRIP_HOTEL_MAP,
        });

        const [checkinDateText, checkoutDateText] = getFormattedDatesForPeriod(
            checkinDate,
            checkoutDate,
        );

        return (
            <div>
                <Text>
                    {insertJSXIntoKey(i18nBlock.crossSaleSubTitle)({
                        hotelsCount: totalHotelCount,
                        checkinDate: (
                            <span
                                {...prepareQaAttributes({
                                    parent: rootQa,
                                    current: 'checkinDate',
                                })}
                            >
                                {checkinDateText}
                            </span>
                        ),
                        checkoutDate: (
                            <span
                                {...prepareQaAttributes({
                                    parent: rootQa,
                                    current: 'checkoutDate',
                                })}
                            >
                                {checkoutDateText}
                            </span>
                        ),
                    })}
                </Text>

                <div className={cx('mapContainer')}>
                    <HotelsCrossSaleMap
                        className={cx('crossSaleMap')}
                        mapClassName={cx('map')}
                        crossSaleData={crossSale}
                        linkUrl={searchUrl}
                        handleLinkClick={handleCrossSaleClick}
                        {...prepareQaAttributes({
                            parent: rootQa,
                            current: 'map',
                        })}
                    />
                </div>
            </div>
        );
    }, [
        isError,
        checkinDate,
        checkoutDate,
        data,
        isLoading,
        request.adults,
        request.childrenAges,
        rootQa,
        handleCrossSaleClick,
    ]);

    useEffect(() => {
        fetch(request);

        reachGoal(ETripsGoal.HOTELS_CROSS_SALE_SHOW);

        return (): void => {
            reset();
        };
    }, [fetch, request, reset]);

    if (!content) {
        return null;
    }

    return (
        <RootComponent
            className={cx('root', deviceMods('root', deviceType), className)}
            {...prepareQaAttributes(rootQa)}
        >
            <Heading level={2}>
                <Flex
                    inline
                    between={4}
                    alignItems="flex-start"
                    flexWrap="nowrap"
                >
                    <span
                        className={cx('title')}
                        {...prepareQaAttributes({
                            parent: rootQa,
                            current: 'title',
                        })}
                    >
                        {title}
                    </span>

                    <CloseIcon
                        className={cx('closeIcon')}
                        onClick={handleHide}
                        {...prepareQaAttributes({
                            parent: rootQa,
                            current: 'closeIcon',
                        })}
                    />
                </Flex>
            </Heading>

            <div className={cx('content')}>{content}</div>
        </RootComponent>
    );
};

export default React.memo(HotelCrossSaleBlock);
