import React, {useCallback} from 'react';
import {stubTrue} from 'lodash';

import ITripHotelOrder from 'types/trips/ITripHotelOrder';
import {IWithClassName} from 'types/withClassName';
import ETripImageType from 'projects/trips/components/TripImage/types/ETripImageType';
import {ETripsGoal} from 'utilities/metrika/types/goals/trips';

import {useReachGoal} from 'utilities/metrika/useReachGoal';
import {
    IWithQaAttributes,
    prepareQaAttributes,
    TQaValue,
} from 'utilities/qaAttributes/qaAttributes';
import {CHAR_NBSP} from 'utilities/strings/charCodes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import getOrderIsCancelled from 'projects/account/pages/TripPage/utilities/getOrderIsCancelled';

import * as i18nBlock from 'i18n/account-TripPage';

import OrdersBlock from 'projects/account/pages/TripPage/components/OrdersBlock/OrdersBlock';
import TripImage from 'projects/trips/components/TripImage/TripImage';
import OrderMainInfo from 'projects/account/pages/TripPage/components/OrderMainInfo/OrderMainInfo';
import SupportAction from 'projects/account/pages/TripPage/components/HotelOrdersBlock/components/SupportAction/SupportAction';
import HotelStars from 'components/HotelStars/HotelStars';

interface IHotelOrdersBlockProps extends IWithClassName, IWithQaAttributes {
    orders: ITripHotelOrder[];
}

const HotelOrdersBlock: React.FC<IHotelOrdersBlockProps> = props => {
    const {className, orders} = props;

    const deviceType = useDeviceType();
    const onOrderNavigate = useReachGoal(ETripsGoal.HOTEL_ORDER_CLICK);

    const renderImage = useCallback((order: ITripHotelOrder) => {
        const isCancelled = getOrderIsCancelled(order);

        return (
            <TripImage
                src={order.image}
                type={ETripImageType.SMALL_SQUARE}
                enableGray={isCancelled}
            />
        );
    }, []);

    const renderOrderMainInfo = useCallback(
        (order: ITripHotelOrder, orderQa?: TQaValue) => {
            const isCancelled = getOrderIsCancelled(order);

            return (
                <OrderMainInfo
                    title={
                        <>
                            {order.title}
                            {order.stars && (
                                <>
                                    {CHAR_NBSP}
                                    <HotelStars
                                        stars={order.stars}
                                        size={deviceType.isMobile ? '8' : '12'}
                                    />
                                </>
                            )}
                        </>
                    }
                    displayDateForward={order.displayCheckinCheckoutDates}
                    isCancelled={isCancelled}
                    {...prepareQaAttributes({
                        parent: orderQa,
                        current: 'orderMainInfo',
                    })}
                />
            );
        },
        [deviceType.isMobile],
    );
    const renderOrderBottom = useCallback(
        (order: ITripHotelOrder, orderQa?: TQaValue) => (
            <SupportAction
                {...prepareQaAttributes({
                    parent: orderQa,
                    current: 'supportAction',
                })}
            />
        ),
        [],
    );

    return (
        <OrdersBlock
            className={className}
            orders={orders}
            title={i18nBlock.hotelReservations({count: orders.length})}
            renderImage={renderImage}
            renderOrderMainInfo={renderOrderMainInfo}
            renderOrderBottom={renderOrderBottom}
            shouldRenderOrderBottom={stubTrue}
            onOrderNavigate={onOrderNavigate}
            {...prepareQaAttributes(props)}
        />
    );
};

export default React.memo(HotelOrdersBlock);
