import React, {useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import genitivePlural from 'utilities/plural/genitivePlural';

import * as i18n from 'i18n/account-TripPage';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';

import cx from './OrderMainInfo.scss';

interface IOrderMainInfoProps extends IWithClassName, IWithQaAttributes {
    title: React.ReactNode;
    displayDateForward: string;
    displayDateBackward?: string;
    refundedTicketsCount?: number;
    isCancelled: boolean;
}

const OrderMainInfo: React.FC<IOrderMainInfoProps> = props => {
    const {
        className,
        title,
        displayDateForward,
        displayDateBackward,
        refundedTicketsCount,
        isCancelled,
    } = props;

    const {isMobile} = useDeviceType();

    const isDirectionDisplayed = Boolean(displayDateBackward);

    const stateText = useMemo(() => {
        if (isCancelled) {
            return typeof refundedTicketsCount === 'number'
                ? i18n.fullRefundIssued()
                : i18n.orderCancelled();
        }

        if (refundedTicketsCount) {
            return i18n.refundSomeTickets({
                displayCount: genitivePlural(refundedTicketsCount),
                isSingle: refundedTicketsCount === 1,
            });
        }

        return null;
    }, [isCancelled, refundedTicketsCount]);

    return (
        <Flex
            className={cx('root', {root_cancelled: isCancelled}, className)}
            flexDirection="column"
            {...prepareQaAttributes(props)}
        >
            <Text
                size={isMobile ? 'm' : 'l'}
                weight="bold"
                color="current"
                {...prepareQaAttributes({
                    parent: props,
                    current: 'title',
                })}
            >
                {title}
            </Text>

            <Text color="current">
                <span
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'dateForward',
                    })}
                >
                    {displayDateForward}
                </span>

                {isDirectionDisplayed && (
                    <span
                        className={cx('direction')}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'directionForward',
                        })}
                    >
                        {i18n.forward()}
                    </span>
                )}
            </Text>

            {displayDateBackward && (
                <Text color="current">
                    <span
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'dateBackward',
                        })}
                    >
                        {displayDateBackward}
                    </span>

                    {isDirectionDisplayed && (
                        <span
                            className={cx('direction')}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'directionBackward',
                            })}
                        >
                            {i18n.backward()}
                        </span>
                    )}
                </Text>
            )}

            {stateText && (
                <Text
                    color={isCancelled ? 'primary' : 'secondary'}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'cancelCaption',
                    })}
                >
                    {stateText}
                </Text>
            )}
        </Flex>
    );
};

export default React.memo(OrderMainInfo);
