import React from 'react';

import typedReactMemo from 'types/typedReactMemo';
import {IWithClassName} from 'types/withClassName';
import {TTripOrder} from 'types/trips/TTripOrder';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
    TQaValue,
} from 'utilities/qaAttributes/qaAttributes';

import Text from 'components/Text/Text';
import Flex from 'components/Flex/Flex';
import Order from 'projects/account/pages/TripPage/components/OrdersBlock/components/Order/Order';

import cx from './OrdersBlock.scss';

interface IOrdersBlockProps<Order extends TTripOrder>
    extends IWithClassName,
        IWithQaAttributes {
    orders: Order[];
    title: string;
    renderImage(order: Order, orderQa?: TQaValue): React.ReactNode;
    renderOrderMainInfo(order: Order, orderQa?: TQaValue): React.ReactNode;
    renderOrderBottom(order: Order, orderQa?: TQaValue): React.ReactNode;
    shouldRenderOrderBottom?(order: Order): boolean;
    onOrderNavigate?(): void;
}

function OrdersBlock<Order extends TTripOrder>(
    props: IOrdersBlockProps<Order>,
): React.ReactElement {
    const {
        className,
        orders,
        title,
        renderImage,
        renderOrderMainInfo,
        renderOrderBottom,
        shouldRenderOrderBottom,
        onOrderNavigate,
    } = props;

    const {isMobile} = useDeviceType();

    return (
        <div className={className} {...prepareQaAttributes(props)}>
            <Text
                className={cx('title')}
                size="xl"
                weight="bold"
                {...prepareQaAttributes({parent: props, current: 'title'})}
            >
                {title}
            </Text>

            <Flex
                className={cx('orders')}
                flexDirection="column"
                between={isMobile ? 8 : 3}
            >
                {orders.map(order => {
                    return (
                        <Order
                            key={order.id}
                            className={cx('order')}
                            order={order}
                            renderImage={renderImage}
                            renderOrderMainInfo={renderOrderMainInfo}
                            renderOrderBottom={renderOrderBottom}
                            shouldRenderOrderBottom={shouldRenderOrderBottom}
                            onNavigate={onOrderNavigate}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'order',
                                key: order.id,
                            })}
                        />
                    );
                })}
            </Flex>
        </div>
    );
}

export default typedReactMemo(OrdersBlock);
