import React, {useMemo} from 'react';

import {TTripOrder} from 'types/trips/TTripOrder';
import {IWithClassName} from 'types/withClassName';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
    TQaValue,
} from 'utilities/qaAttributes/qaAttributes';
import {accountURLs} from 'projects/account/utilities/urls';
import {isConfirmedOrder} from 'projects/trips/utilities/isOrder';

import Separator from 'components/Separator/Separator';
import Link from 'components/Link/Link';
import Card from 'components/Card/Card';
import ArrowRightIcon from 'icons/24/ArrowRight';
import Flex from 'components/Flex/Flex';
import Intersperse from 'components/Intersperse/Intersperse';

import cx from './Order.scss';

interface IOrderProps<Order extends TTripOrder>
    extends IWithClassName,
        IWithQaAttributes {
    order: Order;
    renderImage(order: Order, orderQa?: TQaValue): React.ReactNode;
    renderOrderMainInfo(order: Order, orderQa?: TQaValue): React.ReactNode;
    renderOrderBottom(order: Order, orderQa?: TQaValue): React.ReactNode;
    shouldRenderOrderBottom?(order: Order): boolean;
    onNavigate?(): void;
}

function Order<Order extends TTripOrder>(
    props: IOrderProps<Order>,
): React.ReactElement {
    const {
        className,
        order,
        renderImage,
        renderOrderMainInfo,
        renderOrderBottom,
        shouldRenderOrderBottom = isConfirmedOrder,
        onNavigate,
    } = props;

    const rootQa = getQa(props);

    const deviceType = useDeviceType();

    const {isMobile} = deviceType;

    const image = useMemo(
        () => renderImage(order, rootQa),
        [order, renderImage, rootQa],
    );
    const orderMainInfo = useMemo(
        () => renderOrderMainInfo(order, rootQa),
        [order, renderOrderMainInfo, rootQa],
    );
    const bottomContent = useMemo(() => {
        if (!shouldRenderOrderBottom(order)) {
            return null;
        }

        return renderOrderBottom(order, rootQa);
    }, [order, renderOrderBottom, rootQa, shouldRenderOrderBottom]);

    const content = useMemo(() => {
        return (
            <>
                <Intersperse
                    separator={<Separator margin={isMobile ? 3 : 4} />}
                >
                    <Flex between={4} inline flexWrap="nowrap">
                        <div className={cx('image')}>{image}</div>

                        <div className={cx('orderMainInfo')}>
                            {orderMainInfo}
                        </div>

                        {!isMobile && (
                            <ArrowRightIcon className={cx('goToIcon')} />
                        )}
                    </Flex>
                    {bottomContent}
                </Intersperse>
                <Link
                    className={cx('orderLink')}
                    to={accountURLs.getOrderUrl(order.id)}
                    onClick={onNavigate}
                    {...prepareQaAttributes({
                        parent: rootQa,
                        current: 'orderLink',
                    })}
                />
            </>
        );
    }, [
        bottomContent,
        image,
        isMobile,
        onNavigate,
        order.id,
        rootQa,
        orderMainInfo,
    ]);

    return (
        <div
            className={cx('root', deviceMods('root', deviceType), className)}
            {...prepareQaAttributes(rootQa)}
        >
            {deviceType.isMobile ? (
                content
            ) : (
                <Card
                    className={cx('card', {
                        card_hasBottomContent: Boolean(bottomContent),
                    })}
                    border="thin"
                    withHover
                >
                    {content}
                </Card>
            )}
        </div>
    );
}

export default React.memo(Order);
