import React, {useCallback} from 'react';

import {IWithClassName} from 'types/withClassName';
import ITripTrainOrder from 'types/trips/ITripTrainOrder';
import {ETripsGoal} from 'utilities/metrika/types/goals/trips';

import {useReachGoal} from 'utilities/metrika/useReachGoal';
import {
    IWithQaAttributes,
    prepareQaAttributes,
    TQaValue,
} from 'utilities/qaAttributes/qaAttributes';
import getOrderIsCancelled from 'projects/account/pages/TripPage/utilities/getOrderIsCancelled';

import * as i18nBlock from 'i18n/account-TripPage';

import OrdersBlock from 'projects/account/pages/TripPage/components/OrdersBlock/OrdersBlock';
import Logo from 'projects/account/pages/TripPage/components/TrainOrdersBlock/components/Logo/Logo';
import OrderMainInfo from 'projects/account/pages/TripPage/components/OrderMainInfo/OrderMainInfo';
import DescriptionAndActions from 'projects/account/pages/TripPage/components/TrainOrdersBlock/components/DescriptionAndActions/DescriptionAndActions';

interface ITrainOrdersBlockProps extends IWithClassName, IWithQaAttributes {
    orders: ITripTrainOrder[];
}

const TrainOrdersBlock: React.FC<ITrainOrdersBlockProps> = props => {
    const {className, orders} = props;

    const onOrderNavigate = useReachGoal(ETripsGoal.TRAIN_ORDER_CLICK);

    const renderImage = useCallback((order: ITripTrainOrder) => {
        const isCancelled = getOrderIsCancelled(order);

        return <Logo isCancelled={isCancelled} />;
    }, []);

    const renderOrderMainInfo = useCallback(
        (order: ITripTrainOrder, orderQa?: TQaValue) => {
            const isCancelled = getOrderIsCancelled(order);

            return (
                <OrderMainInfo
                    isCancelled={isCancelled}
                    {...order}
                    {...prepareQaAttributes({
                        parent: orderQa,
                        current: 'orderMainInfo',
                    })}
                />
            );
        },
        [],
    );

    const renderOrderBottom = useCallback(
        (order: ITripTrainOrder, orderQa?: TQaValue) => {
            return (
                <DescriptionAndActions
                    order={order}
                    {...prepareQaAttributes({
                        parent: orderQa,
                        current: 'descriptionAndActions',
                    })}
                />
            );
        },
        [],
    );

    return (
        <OrdersBlock
            className={className}
            orders={orders}
            title={i18nBlock.trainTickets({count: orders.length})}
            renderImage={renderImage}
            renderOrderMainInfo={renderOrderMainInfo}
            renderOrderBottom={renderOrderBottom}
            onOrderNavigate={onOrderNavigate}
            {...prepareQaAttributes(props)}
        />
    );
};

export default React.memo(TrainOrdersBlock);
