import React, {useCallback} from 'react';

import ITripTrainOrder from 'types/trips/ITripTrainOrder';
import {ORDER_TYPE_VALUES} from 'projects/account/lib/orders/types';
import {ETripsGoal} from 'utilities/metrika/types/goals/trips';

import getDownloadFilename from 'projects/account/utilities/downloadBlank/getDownloadFilename';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import getDownloadBlankUrl from 'projects/account/pages/Order/utilities/downloadBlank/getDownloadBlankUrl';
import {usePrint} from 'utilities/hooks/usePrint';
import {reachGoal} from 'utilities/metrika';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useReachGoal} from 'utilities/metrika/useReachGoal';

import * as i18nBlock from 'i18n/account-TripPage';

import Flex from 'components/Flex/Flex';
import TrainDescription from 'projects/account/pages/TripPage/components/TrainOrdersBlock/components/DescriptionAndActions/components/TrainDescription/TrainDescription';
import DownloadIcon from 'icons/16/Download';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import Button from 'components/Button/Button';
import PrintIcon from 'icons/16/Print';
import ButtonLink from 'components/ButtonLink/ButtonLink';

import cx from './DescriptionAndActions.scss';

interface IDescriptionAndActionsProps extends IWithQaAttributes {
    order: ITripTrainOrder;
}

const DescriptionAndActions: React.FC<IDescriptionAndActionsProps> = props => {
    const {order} = props;

    const {isMobile, isDesktop} = useDeviceType();

    const downloadUrl = getDownloadBlankUrl(
        {id: order.id},
        ORDER_TYPE_VALUES.TRAINS_GENERIC,
    );
    const {isPrinting, startPrinting} = usePrint({
        printable: downloadUrl,
        type: 'pdf',
    });

    const handlePrintClick = useCallback(() => {
        startPrinting();

        reachGoal(ETripsGoal.TRAIN_ORDER_PRINT_BUTTON_CLICK);
    }, [startPrinting]);

    const handleDownloadClick = useReachGoal(
        ETripsGoal.TRAIN_ORDER_DOWNLOAD_BUTTON_CLICK,
    );

    return (
        <Flex
            className={cx('root')}
            inline
            flexWrap="nowrap"
            {...prepareQaAttributes(props)}
        >
            <TrainDescription
                className={cx('trainDescription')}
                trains={order.trains}
                hasTransferWithStationChange={
                    order.hasTransferWithStationChange
                }
                {...prepareQaAttributes({
                    parent: props,
                    current: 'trainDescription',
                })}
            />

            <ButtonLink
                className={cx('downloadButton')}
                shape={isMobile ? 'circle' : undefined}
                size="l"
                onClick={handleDownloadClick}
                url={downloadUrl}
                download={getDownloadFilename('order', order.id, 'pdf')}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'downloadButton',
                })}
            >
                {isMobile ? (
                    <DownloadIcon />
                ) : (
                    <TextWithIcon
                        text={i18nBlock.download()}
                        iconLeft={DownloadIcon}
                        iconLeftClassName={cx('downloadIcon')}
                    />
                )}
            </ButtonLink>

            {isDesktop && (
                <Button
                    className={cx('printButton')}
                    size="l"
                    icon={<PrintIcon />}
                    disabled={isPrinting}
                    onClick={handlePrintClick}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'printButton',
                    })}
                />
            )}
        </Flex>
    );
};

export default React.memo(DescriptionAndActions);
