import React from 'react';

import {ITrain} from 'types/trips/ITripTrainOrder';
import {IWithClassName} from 'types/withClassName';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/account-TripPage';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import ConnectionIcon from 'icons/16/Connection';

import cx from './TrainDescription.scss';

interface ITrainDescriptionProps extends IWithClassName, IWithQaAttributes {
    trains: ITrain[];
    hasTransferWithStationChange: boolean;
}

const TrainDescription: React.FC<ITrainDescriptionProps> = props => {
    const {className, trains, hasTransferWithStationChange} = props;

    const rootClassName = cx('root', className);

    if (!trains.length) {
        return null;
    }

    if (hasTransferWithStationChange && trains.length > 1) {
        return (
            <Flex
                className={rootClassName}
                flexDirection="column"
                {...prepareQaAttributes(props)}
            >
                <Text
                    className={cx('textOneLine')}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'combinedDescription',
                    })}
                >
                    {i18nBlock.trainsDescription({
                        train1: trains[0].number,
                        train2: trains[1].number,
                    })}
                </Text>

                <TextWithIcon
                    iconLeft={ConnectionIcon}
                    iconLeftClassName={cx('transferIcon')}
                    text={i18nBlock.transferWithStationChange()}
                    textClassName={cx('transferText', 'textOneLine')}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'transferText',
                    })}
                />
            </Flex>
        );
    }

    return (
        <Flex
            className={rootClassName}
            flexDirection="column"
            {...prepareQaAttributes(props)}
        >
            {trains.map(({description}, index) => {
                return (
                    <Text
                        key={index}
                        className={
                            trains.length > 1 ? cx('textOneLine') : undefined
                        }
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'trainDescription',
                            key: index,
                        })}
                    >
                        {description}
                    </Text>
                );
            })}
        </Flex>
    );
};

export default React.memo(TrainDescription);
