import {FC, memo, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import ETripImageType from 'projects/trips/components/TripImage/types/ETripImageType';

import {CustomDispatch} from 'reducers/trains/customDispatch';
import {clearTrip} from 'reducers/trips/tripPage/actions';

import tripPageSelector from 'projects/account/pages/TripPage/components/TripContent/selectors/tripContentSelector';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {
    isActivitiesBlock,
    isAviaOrderBlock,
    isBusOrderBlock,
    isForecastBlock,
    isHotelCrossSaleBlock,
    isHotelOrderBlock,
    isNotificationsBlock,
    isRestrictionsBlock,
    isTrainOrderBlock,
} from 'projects/account/pages/TripPage/utilities/blocks';
import useDispatchedAction from 'utilities/hooks/useDispatchedAction';
import {deviceMods} from 'utilities/stylesUtils';

import TripError from 'projects/account/pages/TripPage/components/TripError/TripError';
import Skeleton from 'projects/account/pages/TripPage/components/Skeleton/Skeleton';
import DesktopSupportPhone from 'projects/trips/components/DesktopSupportPhone/DesktopSupportPhone';
import Flex from 'components/Flex/Flex';
import BackLink from 'projects/account/pages/TripPage/components/BackLink/BackLink';
import HotelCrossSaleBlock from 'projects/account/pages/TripPage/components/HotelCrossSaleBlock/HotelCrossSaleBlock';
import Heading from 'components/Heading/Heading';
import TripImage from 'projects/trips/components/TripImage/TripImage';
import NotificationsBlock from 'projects/account/pages/TripPage/components/NotificationsBlock/NotificationsBlock';
import AviaOrdersBlock from 'projects/account/pages/TripPage/components/AviaOrdersBlock/AviaOrdersBlock';
import TrainOrdersBlock from 'projects/account/pages/TripPage/components/TrainOrdersBlock/TrainOrdersBlock';
import HotelOrdersBlock from 'projects/account/pages/TripPage/components/HotelOrdersBlock/HotelOrdersBlock';
import BusOrdersBlock from 'projects/account/pages/TripPage/components/BusOrdersBlock/BusOrdersBlock';
import ActivitiesBlock from 'projects/account/pages/TripPage/components/ActivitiesBlock/ActivitiesBlock';
import RestrictionsBlock from 'projects/account/pages/TripPage/components/RestrictionsBlock/RestrictionsBlock';
import ForecastBlock from 'projects/account/pages/TripPage/components/ForecastBlock/ForecastBlock';

import cx from './TripContent.scss';

interface ITripContentProps extends IWithQaAttributes {}

const TripContent: FC<ITripContentProps> = props => {
    const rootQa = getQa(props);

    const dispatch: CustomDispatch = useDispatch();
    const deviceType = useDeviceType();
    const {trip, hiddenHotelCrossSaleBlocks} = useSelector(tripPageSelector);

    const clearError = useDispatchedAction(clearTrip, dispatch);

    const {isDesktop} = deviceType;

    const backLink = useMemo(() => {
        return (
            <BackLink
                className={cx('backLink')}
                {...prepareQaAttributes({parent: rootQa, current: 'backLink'})}
            />
        );
    }, [rootQa]);

    const header = useMemo(
        () => (
            <Flex
                between={6}
                inline
                flexWrap="nowrap"
                alignItems="baseline"
                justifyContent="space-between"
            >
                {backLink}

                {isDesktop && (
                    <DesktopSupportPhone
                        className={cx('supportPhone')}
                        {...prepareQaAttributes({
                            parent: rootQa,
                            current: 'desktopSupportPhone',
                        })}
                    />
                )}
            </Flex>
        ),
        [backLink, isDesktop, rootQa],
    );

    const content = useMemo(() => {
        if (trip.isFailed) {
            return (
                <Flex className={cx('mainContent')} flexDirection="column">
                    {header}

                    <TripError
                        className={cx('errorContent')}
                        status={trip.status}
                        onNavigateToIndex={clearError}
                    />
                </Flex>
            );
        }

        if (trip.isLoading) {
            return (
                <div className={cx('mainContent')}>
                    {header}

                    <Skeleton
                        className={cx('skeleton')}
                        {...prepareQaAttributes({
                            parent: rootQa,
                            current: 'skeleton',
                        })}
                    />
                </div>
            );
        }

        if (!trip.value) {
            return null;
        }

        const {id, title, image, blocks} = trip.value;

        const hotelCrossSaleBlock = blocks.find(isHotelCrossSaleBlock);
        const activitiesBlock = blocks.find(isActivitiesBlock);
        const notificationsBlock = blocks.find(isNotificationsBlock);
        const restrictionsBlock = blocks.find(isRestrictionsBlock);
        const forecastBlock = blocks.find(isForecastBlock);

        const titleNode = (
            <Heading className={cx('title')} level={1}>
                {title}
            </Heading>
        );

        return (
            <>
                <div className={cx('mainContent')}>
                    <Flex flexDirection="column">
                        {isDesktop ? (
                            header
                        ) : (
                            <>
                                {backLink}
                                {titleNode}
                            </>
                        )}

                        <div className={cx('imageWrapper')}>
                            <TripImage
                                className={cx('image')}
                                type={ETripImageType.LARGE_SQUARE}
                                src={image ?? undefined}
                                enableFog={isDesktop}
                                {...prepareQaAttributes({
                                    parent: rootQa,
                                    current: 'tripImage',
                                })}
                            />
                            <div className={cx('imageContentWrapper')}>
                                <Flex
                                    className={cx('imageContent')}
                                    flexDirection="column"
                                    between={2}
                                >
                                    {isDesktop && titleNode}
                                    {forecastBlock && (
                                        <ForecastBlock
                                            forecastBlock={forecastBlock}
                                            {...prepareQaAttributes({
                                                parent: rootQa,
                                                current: 'forecastBlock',
                                            })}
                                        />
                                    )}
                                </Flex>
                            </div>
                        </div>

                        {notificationsBlock && (
                            <NotificationsBlock
                                className={cx('notificationsBlock')}
                                notificationsBlock={notificationsBlock}
                            />
                        )}

                        {restrictionsBlock && (
                            <RestrictionsBlock
                                className={cx('restrictionsBlock')}
                                restrictionsBlock={restrictionsBlock}
                            />
                        )}

                        <div className={cx('mainBlocks')}>
                            {blocks.map((block, index) => {
                                if (
                                    isAviaOrderBlock(block) &&
                                    block.orders.length > 0
                                ) {
                                    return (
                                        <AviaOrdersBlock
                                            key={index}
                                            className={cx('ordersBlock')}
                                            orders={block.orders}
                                            {...prepareQaAttributes({
                                                parent: rootQa,
                                                current: 'aviaOrdersBlock',
                                            })}
                                        />
                                    );
                                }

                                if (
                                    isTrainOrderBlock(block) &&
                                    block.orders.length > 0
                                ) {
                                    return (
                                        <TrainOrdersBlock
                                            key={index}
                                            className={cx('ordersBlock')}
                                            orders={block.orders}
                                            {...prepareQaAttributes({
                                                parent: rootQa,
                                                current: 'trainOrdersBlock',
                                            })}
                                        />
                                    );
                                }

                                if (
                                    isHotelOrderBlock(block) &&
                                    block.orders.length > 0
                                ) {
                                    return (
                                        <HotelOrdersBlock
                                            key={index}
                                            className={cx('ordersBlock')}
                                            orders={block.orders}
                                            {...prepareQaAttributes({
                                                parent: rootQa,
                                                current: 'hotelOrdersBlock',
                                            })}
                                        />
                                    );
                                }

                                if (
                                    isBusOrderBlock(block) &&
                                    block.orders.length > 0
                                ) {
                                    return (
                                        <BusOrdersBlock
                                            key={index}
                                            className={cx('ordersBlock')}
                                            orders={block.orders}
                                            {...prepareQaAttributes({
                                                parent: rootQa,
                                                current: 'busOrdersBlock',
                                            })}
                                        />
                                    );
                                }

                                return null;
                            })}
                        </div>
                    </Flex>
                </div>

                {hotelCrossSaleBlock &&
                    !hiddenHotelCrossSaleBlocks.includes(id) && (
                        <HotelCrossSaleBlock
                            className={cx('crossSaleBlock')}
                            crossSale={hotelCrossSaleBlock.block}
                            tripId={id}
                            {...prepareQaAttributes({
                                parent: rootQa,
                                current: 'hotelsCrossSaleBlock',
                            })}
                        />
                    )}

                {activitiesBlock && (
                    <ActivitiesBlock
                        className={cx('activitiesBlock')}
                        activitiesBlock={activitiesBlock}
                        {...prepareQaAttributes({
                            parent: rootQa,
                            current: 'activitiesBlock',
                        })}
                    />
                )}
            </>
        );
    }, [
        backLink,
        clearError,
        header,
        hiddenHotelCrossSaleBlocks,
        isDesktop,
        rootQa,
        trip.isFailed,
        trip.isLoading,
        trip.status,
        trip.value,
    ]);

    return (
        <Flex
            className={cx('root', deviceMods('root', deviceType))}
            flexDirection="column"
        >
            {content}
        </Flex>
    );
};

export default memo(TripContent);
