import {FC, memo, useMemo} from 'react';

import {URLs} from 'constants/urls';

import {EHttpStatusCodes} from 'types/httpStatusCodes';
import {IWithClassName} from 'types/withClassName';

import {
    EActionType,
    getErrorRenderInfo,
} from 'projects/account/pages/TripPage/components/TripError/utilities/getErrorRenderInfo';
import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useUserInfo} from 'utilities/hooks/useUserInfo';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';
import {EPassportMode, getPassportUrl} from 'utilities/url/getPassportUrl';

import * as i18n from 'i18n/account-TripPage';
import * as i18nTrips from 'i18n/trips-common';

import Flex from 'components/Flex/Flex';
import Error500M from 'icons/illustrations/Error500/Error500M';
import Heading from 'components/Heading/Heading';
import Text from 'components/Text/Text';
import {IButtonProps} from 'components/Button/Button';
import ButtonLink from 'components/ButtonLink/ButtonLink';
import useReturnPathUrl from 'components/User/hooks/useReturnPathUrl';
import LoadingErrorModal from 'projects/trips/components/LoadingErrorModal/LoadingErrorModal';

import cx from './TripError.scss';

interface ITripErrorProps extends IWithClassName {
    status: number | null;
    onNavigateToIndex(): void;
}

const TripError: FC<ITripErrorProps> = props => {
    const {className, status, onNavigateToIndex} = props;

    const deviceType = useDeviceType();
    const userInfo = useUserInfo();
    const returnPathUrl = useReturnPathUrl();

    const isAuth = isAuthUser(userInfo);
    const {isMobile} = deviceType;

    const errorRenderInfo = useMemo(() => {
        return getErrorRenderInfo(status, isAuth);
    }, [isAuth, status]);

    const actionButton = useMemo(() => {
        if (!errorRenderInfo) {
            return null;
        }

        const commonProps: IButtonProps = {
            className: cx('action'),
            theme: 'primary',
            size: isMobile ? 'l' : 'm-inset',
        };

        const loginUrl = getPassportUrl({
            mode: EPassportMode.AUTH,
            passportHost: userInfo.passportPath,
            // Хотим отправлять сразу на список поездок, если поездки не существует
            retpath:
                status === EHttpStatusCodes.NotFound
                    ? `${location.origin}${URLs.accountTrips}`
                    : returnPathUrl,
        });

        if (errorRenderInfo.actionType === EActionType.LOGIN) {
            return (
                <ButtonLink {...commonProps} url={loginUrl}>
                    {i18n.errorActionLogin()}
                </ButtonLink>
            );
        }

        if (errorRenderInfo.actionType === EActionType.RE_LOGIN) {
            return (
                <ButtonLink
                    {...commonProps}
                    url={
                        isAuth
                            ? getPassportUrl({
                                  mode: EPassportMode.LOGOUT,
                                  passportHost: userInfo.passportPath,
                                  yu: userInfo.yu,
                                  uid: userInfo.uid,
                                  retpath: loginUrl,
                              })
                            : undefined
                    }
                >
                    {i18n.errorActionReLogin()}
                </ButtonLink>
            );
        }

        if (errorRenderInfo.actionType === EActionType.VIEW_TRIPS) {
            return (
                <ButtonLink {...commonProps} to={URLs.accountTrips}>
                    {i18n.errorActionViewTrips()}
                </ButtonLink>
            );
        }
    }, [errorRenderInfo, isAuth, isMobile, returnPathUrl, status, userInfo]);

    // Неизвестная ошибка
    if (!errorRenderInfo) {
        return (
            <LoadingErrorModal
                isVisible
                onNavigateToIndex={onNavigateToIndex}
            />
        );
    }

    return (
        <Flex
            className={cx('root', deviceMods('root', deviceType), className)}
            flexDirection="column"
            alignItems="center"
            justifyContent="center"
        >
            <Error500M className={cx('illustration')} />

            <Heading className={cx('title')} level={2}>
                {i18nTrips.errorTitle()}
            </Heading>

            <Text className={cx('description')}>
                {errorRenderInfo.description}
            </Text>

            {actionButton}
        </Flex>
    );
};

export default memo(TripError);
